//-*-c++-*-
#ifndef INCLUDED_BehaviorSwitchControl_h_
#define INCLUDED_BehaviorSwitchControl_h_

#include "MenuControl.h"
#include "Behaviors/BehaviorBase.h"
#include "Shared/ReferenceCounter.h"
#include "Shared/Factory.h"
#include "Shared/debuget.h"

//! Holds some utility classes and functions for BehaviorSwitchControl which shouldn't be stored in a templated class
class BehaviorSwitchControlBase : public MenuControl {
public:
	//! A simple utility class to allow the BehaviorSwitchControl's to be able to deactivate the current behavior when a new one becomes active
	/*! Most behaviors are either major actions which you'll only want one of active
	 *  at a time, or else their background monitors of some sort, that can run in different
	 *  combinations.  Think radio buttons vs. checkboxes.  This will help you implement the
	 *  "radio button" style... just assign all the behaviors to the same group, they will
	 *  automatically use it to turn the previous behavior off when a new one becomes active.\n
	 *  Pass NULL instead of one of these to get checkbox-style. */
	class BehaviorGroup : public ReferenceCounter {
	public:
		BehaviorGroup() : curBehavior(NULL) { } //!< contructor
		~BehaviorGroup() { if(curBehavior!=NULL) curBehavior->DoStop(); } //!< destructor, will stop the current behavior if it was a one-shot
		BehaviorBase * curBehavior; //!< pointer to current behavior
	private:
		BehaviorGroup(const BehaviorGroup&); //!< shouldn't be called
		BehaviorGroup operator=(const BehaviorGroup&); //!< shouldn't be called
	};
	
	//! constructor
	BehaviorSwitchControlBase(const std::string& n, BehaviorGroup* bg, bool retain)
		: MenuControl(n), retained(true), behgrp(bg) {
		retained=retain;
		if(behgrp!=NULL)
			behgrp->AddReference();
	}

	//! destructor
	virtual ~BehaviorSwitchControlBase() {
		if(behgrp!=NULL)
			behgrp->RemoveReference();
	}

	//! activates the behavior, handy for making start-up behaviors that you can turn off again with the Controller
	/*! If you start twice without stopping (ie it's already running), shouldn't do anything */
	virtual BehaviorSwitchControlBase* start()=0;
	virtual BehaviorSwitchControlBase* stop()=0; //!< stops the behavior
	virtual BehaviorSwitchControlBase* toggle()=0; //!< toggles the behavior

	//! tells the current behavior (if there is one) to stop then loads its own
	/*! @return NULL unless there are submenus */
	virtual ControlBase * activate(MotionManager::MC_ID display) {
		if(slotsSize()==0) {
			toggle();
			return NULL;
		} else
			return MenuControl::activate(display);
	}

protected:
	bool retained; //!< true if the behavior should be generated once and retained after DoStop.  Otherwise, a new one is generated each time it is started
	BehaviorGroup * behgrp; //!< the behavior group this belongs to.  Uses this to track the "current" behavior

private:
	BehaviorSwitchControlBase(const BehaviorSwitchControlBase&); //!< shouldn't copy these
	BehaviorSwitchControlBase operator=(const BehaviorSwitchControlBase&); //!< shouldn't assign these
};




//! Allows proper switching between major behaviors, calling DoStart and DoStop
template < class B, class Al = Factory< B > >
class BehaviorSwitchControl : public BehaviorSwitchControlBase {
public:
	//! constructor, can use this to toggle a single behavior on and off
	BehaviorSwitchControl(const std::string& n, bool retain=false)
		: BehaviorSwitchControlBase(n,NULL,retain), mybeh(NULL) {}
	//! constructor, if you want to use an already constructed behavior, can pass NULL for behavior group
	BehaviorSwitchControl(const std::string& n, B* beh, BehaviorGroup* bg=NULL)
		: BehaviorSwitchControlBase(n,bg,true), mybeh(beh) {
		retained=true;
		if(mybeh!=NULL)
			mybeh->AddReference();
	}
	//! constructor, needs to know what group its in and whether to retain its behavior
	BehaviorSwitchControl(const std::string& n, BehaviorGroup* bg, bool retain=false)
		: BehaviorSwitchControlBase(n,bg,retain), mybeh(NULL) {}
	
	//! destructor
	virtual ~BehaviorSwitchControl() {
		if(mybeh!=NULL && retained)
			mybeh->RemoveReference();
	}

	virtual BehaviorSwitchControl<B,Al>* start() { if(!isRunning()) { stopother(); startmine(); } return this; }

	virtual BehaviorSwitchControl<B,Al>* stop() { if(isRunning()) stopother(); return this; }

	virtual BehaviorSwitchControl<B,Al>* toggle() { if(isRunning()) stopother(); else { stopother(); startmine(); } return this; }

	//! adds a status to the name: - if in memory, # if running
	virtual const char* getName() const {
		if(!isValid())//(mybeh==NULL || !retained && (behgrp==NULL || behgrp->curBehavior!=mybeh))
			return MenuControl::getName();
		else
			return mybeh->getName();
	}
	

protected:

	//! Stops the "other" guy's behavior - if ::behgrp is NULL, stops ourselves
	virtual void stopother() {
		if(behgrp==NULL) {
			if(mybeh!=NULL) {
				if(mybeh->isActive()) {
					mybeh->DoStop();
					if(!retained)
						mybeh=NULL;
				} else
					ASSERT(retained,"null group, non-null not retained beh, not active, did you call inherited DoStart/DoStop in your Behavior?");
			}
		} else if(behgrp->curBehavior!=NULL) {
			behgrp->curBehavior->DoStop();
			behgrp->curBehavior=NULL;
		}
	}

	//! Starts our behavior
	virtual void startmine() {
		if(!retained) {
			mybeh=Al::construct();
			if(behgrp!=NULL)
				behgrp->curBehavior=mybeh;
		} else {
			if(mybeh==NULL) {
				mybeh=Al::construct();
				mybeh->AddReference();
			}
			if(behgrp!=NULL)
				behgrp->curBehavior=mybeh;
		}
		mybeh->DoStart();
	}

	//! Returns true if the associated behavior is running
	virtual bool isRunning() const {
		if(mybeh==NULL) //not created or has been destroyed, definitely not running
			return false;
		// so, beh has been created (but may have been stopped by another in the group)
		if(behgrp==NULL) //no group
			return mybeh->isActive(); //just check active flag (is valid object, we would have set it to NULL if we stopped it ourselves)
		// so, we're in a group, someone else could have stopped us
		return (behgrp->curBehavior==mybeh); //all we can see is if the current behavior is ours.  If it is, it'll be active
	}

	//! Returns true if mybeh is pointing to a valid object
	virtual bool isValid() const {
		if(isRunning())
			return true;
		return retained;
	}

	B* mybeh; //!< used to store the behavior.  If retained and non-NULL, will be valid.  However, if not retained, only valid if equals behgrp->curBehavior

private:
	BehaviorSwitchControl(const BehaviorSwitchControl&); //!< shouldn't call this
	BehaviorSwitchControl operator=(const BehaviorSwitchControl&); //!<shouldn't call this
};

/*! @file
 * @brief Defines BehaviorSwitchControl and the BehaviorSwitch namespace - a control for turning behaviors on and off
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name: tekkotsu-0_95 $
 * $Revision: 1.1 $
 * $State: Exp $
 * $Date: 2003/03/01 20:53:27 $
 */

#endif
