#include "WorldModel.h"

#include <MCOOP.h>
#include "Shared/Util.h"
#include "Vision/Vision.h"
#include "Shared/WorldState.h"
#include "Motion/MMAccessor.h"
#include "Motion/MotionManager.h"
#include "Motion/MotionCommand.h"
#include "Motion/HeadPointerMC.h"
#include "Motion/SharedMotion.h"
#include "Shared/mathutils.h"
#include "Events/VisionEvent.h"
#include "Wireless/Wireless.h"
#include "WorldModelSerializer.h"
#include "Shared/WorldStateSerializer.h"
#include <math.h>

using namespace mathutils;
using namespace VisionEventNS;

WorldModel *wmodel=NULL;

WorldModel::WorldModel() : headpointer_id(MotionManager::invalid_MC_ID), _posX(0.0), _posY(0.0), _sdev(0.0), _bodyOrient(0.0),_adev(0.0)
{
  wmser=new WorldModelSerializer();
  _numMarkers=0;
  _centerX=vision->width/2;
  _centerY=vision->height/2;
  _locState=loc_disabled;
}

const double WorldModel::_marker_x[NUM_MARKERS]=
     { 0.000000,35.000000,60.621778,70.000000,60.621778,35.000000,
       -0.000000,-35.000000,-60.621778,-70.000000,-60.621778,-35.000000 };
const double WorldModel::_marker_y[NUM_MARKERS]=
     { 70.000000,60.621778,35.000000,0.000000,-35.000000,-60.621778,
       -70.000000,-60.621778,-35.000000,0.000000,35.000000,60.621778 };
const double WorldModel::_poolRadius=70.5;
       
void WorldModel::localize()
{
  vision->enableEvents(MarkersSID);
  _locState=loc_enabled;
}

void WorldModel::abortLocalize()
{
  _locState=loc_disabled;
  vision->disableEvents(MarkersSID);
}

void WorldModel::reportMarkers(Marker* markers,int num_markers)
{
  _numMarkers=num_markers;
  for (int i=0; i<num_markers; i++) {
    _markers[i].cen_x=markers[i].cen_x;
    _markers[i].cen_y=markers[i].cen_y;
    _markers[i].marker=markers[i].marker;
  }
}

void WorldModel::initialize()
{
  headpointer_id=motman->addMotion(SharedMotion<HeadPointerMC>(), MotionCommand::kHighPriority);
  MMAccessor<HeadPointerMC>(headpointer_id).mc()->setActive(false);

  for (int i=0; i<30; i++) {
    for (int j=0; j<30; j++) {
      int t=(14.5-i)*(14.5-i)+(14.5-j)*(14.5-j);
      if (t<210) _cells[i][j]=2;
      else if (t<240) _cells[i][j]=3;
      else _cells[i][j]=1;
    }
  }
}

void WorldModel::close()
{
  motman->removeMotion(headpointer_id);
}

void WorldModel::frameProcessor()
{
  slowLocalizationSM();
  //localizationSM();
  _numMarkers=0;
}

void WorldModel::slowLocalizationSM()
{
  static int accuracy[_num_markers]; // distance of marker from center
  static double distance[_num_markers]; // negative=out of range
  static double angle[_num_markers];
  //  static int readings[_num_markers]; // for debug only
  
  switch (_locState) {
    case loc_disabled:
      return;

    case loc_enabled: // move all the way left
      MMAccessor<HeadPointerMC>(headpointer_id).mc()->setActive(true);
      for (int i=0; i<_num_markers;i++) {
        accuracy[i]=100000;
        distance[i]=0.0;
        angle[i]=0.0;
        //readings[i]=0;
      }
      moveHeadAbsRadians(0.0,M_PI/2,0.0);
      _locState=loc_moveto1;
      break;

    case loc_moveto1: // move all the way left
      if (state->outputs[HeadOffset+PanOffset]>8*M_PI/18) {
        _locState=loc_moveto2;
//        printf("scanning ...\n");
      }
      break;
      
    case loc_moveto2: // scan to the right slowly
      if (state->outputs[HeadOffset+PanOffset]<-8*M_PI/18) {
        _locState=loc_done;
      } else {
        for (int i=0; i<_numMarkers; i++) {
          int sd=abs_t(_centerX-_markers[i].cen_x);
          if (sd<accuracy[_markers[i].marker]) {
            accuracy[_markers[i].marker]=sd;
            distance[_markers[i].marker]=state->sensors[IRDistOffset];
            angle[_markers[i].marker]=state->outputs[HeadOffset+PanOffset];
            //readings[_markers[i].marker]++;
          }
        }
        
        moveHeadScreen(_centerX+25,_centerY); 
      }
      break;

    case loc_done:
      vision->disableEvents(MarkersSID);
      if (fitToCircle(accuracy, distance, angle)) {
        wmser->serializePos();
      }
      MMAccessor<HeadPointerMC>(headpointer_id).mc()->setActive(false);
      _locState=loc_disabled;
      break;
  }
}

bool WorldModel::fitToCircle(int accuracy[], double distance[], double angle[])
{
  // move valid readings to beginning of array
  int numValid=0;
  double positions[2*_num_markers];
  int markers[_num_markers];
  for (int i=0; i<_num_markers; i++)
    if (accuracy[i]<8 && distance[i]>=10.0 && distance[i]<900.0) {
      if (numValid<i) {
        distance[numValid]=distance[i];
        angle[numValid]=angle[i];
      }
      markers[numValid]=i;
      numValid++;
    }
  
//  printf("numvalid = %d\n",numValid);
/*  for (int i=0; i<numValid; i++) {
    printf("%f\n", distance[i]);
  }*/

  if (numValid<2) return false;

  int numSamples=0;
  static const int maxSamples=10; // set for performance reasons
  int distThresh=(numValid==2)?1:2;
  
  for (int i=0; i<numValid; i++)
    for (int j=i+1; j<numValid && numSamples<maxSamples; j++)
      if (markerDistance(markers[i],markers[j])>=distThresh) 
        if (triangulate(markers[i],distance[i],markers[j],distance[j],positions,numSamples*2))
          numSamples++;
 
/*  for (int i=0; i<numSamples; i++) {
    printf("%d %f %f\n",i,positions[i*2],positions[i*2+1]);
  }*/
  return (numSamples>0)
         && updatePosition2(numSamples,positions)
         && updateOrientation2(numValid,markers,angle);
}

bool WorldModel::updateOrientation2(int numValid, int markers[], double angle[])
{
  sout->pdebug(0,"numValid: %d, x: %f , y: %f \n",numValid,_posX,_posY);
  for (int i=0; i<numValid; i++) {
    sout->pdebug(0,"%d %f\n",markers[i],angle[i]);
  }
  double orients[_num_markers];
  double headOrient, bodyOrient;
  _bodyOrient=0.0; _adev=0.0;
  double sines=0.0, cosines=0.0;
  for (int i=0; i<numValid; i++) {
    double x2=_marker_x[markers[i]];
    double y2=_marker_y[markers[i]];
    headOrient=atan2(y2-_posY,x2-_posX);
    if (headOrient<0) headOrient+=2*M_PI;  // range from 0 to 2PI
    bodyOrient=headOrient-angle[i];
    if (bodyOrient<0) bodyOrient+=2*M_PI;
    if (bodyOrient>=2*M_PI) bodyOrient-=2*M_PI;
    orients[i]=bodyOrient;
    sines+=sin(bodyOrient);
    cosines+=cos(bodyOrient);
  }
  _bodyOrient=atan2(sines,cosines);
  for (int i=0; i<numValid; i++) {
    double diff=abs_t(orients[i]-_bodyOrient);
    diff=(diff<=180)?diff:360-diff;
    if (diff<0) diff+=2*M_PI;
    if (diff>=2*M_PI) diff-=2*M_PI;
    _adev+=squared(diff);
  }
  _adev=sqrt(_adev/(double)numValid);
  return true;
}
/*
bool WorldModel::updateOrientation2(int numSamples, double positions[], int markers[], double angle[])
{
//  cout << "in updateOrientation2" << endl;
  sout->pdebug(0,"numSamples: %d, x: %f , y: %f \n",numSamples,_posX,_posY);
  for (int i=0; i<numSamples; i++) {
    sout->pdebug(0,"%d %f %f\n",markers[i],positions[i*2], positions[i*2+1],angle[i]);
  }
  double orients[_num_markers];
  double headOrient, bodyOrient;
  _bodyOrient=0.0; _adev=0.0;
  double sines=0.0, cosines=0.0;
  for (int i=0; i<numSamples; i++) {
    double x2=_marker_x[markers[i]];
    double y2=_marker_y[markers[i]];
    headOrient=atan2(y2-positions[i*2+1],x2-positions[i*2]);
    if (headOrient<0) headOrient+=2*M_PI;  // range from 0 to 2PI
    bodyOrient=headOrient-angle[i];
    if (bodyOrient<0) bodyOrient+=2*M_PI;
    if (bodyOrient>=2*M_PI) bodyOrient-=2*M_PI;
    orients[i]=bodyOrient;
    sines+=sin(bodyOrient);
    cosines+=cos(bodyOrient);
  }
  _bodyOrient=atan2(sines,cosines);
  for (int i=0; i<numSamples; i++) {
    double diff=abs_t(orients[i]-_bodyOrient);
    diff=(diff<=180)?diff:360-diff;
    if (diff<0) diff+=2*M_PI;
    if (diff>=2*M_PI) diff-=2*M_PI;
    _adev+=squared(diff);
  }
  _adev=sqrt(_adev/(double)numSamples);
  return true;
}*/

bool WorldModel::updatePosition2(int numSamples, double positions[]) {
  //cout << " in updateposotion2" << endl;
  double x=0.0,y=0.0,sdev=0.0;
  for (int i=0; i<numSamples; i++) {
    x+=positions[i*2];
    y+=positions[i*2+1];
  }
  x=x/numSamples;
  y=y/numSamples;
  for (int i=0; i<numSamples; i++) {
    sdev+=squared(positions[i*2]-x);
    sdev+=squared(positions[i*2+1]-y);
  }
  sdev=sqrt(sdev/(2*numSamples));
  _posX=x;
  _posY=y;
  _sdev=sdev;
//  printf("%d %f %f %f\n",numSamples,_posX,_posY,_sdev); 
  return true;
}

int WorldModel::markerDistance(int marker1, int marker2) {
  int dist=abs_t(marker1-marker2);
  return (dist<=_num_markers/2)?dist:_num_markers-dist;
}

void WorldModel::localizationSM()
{
  static int scanDirection=DIR_LEFT;
  static int numDirSwitches=0;
  static int locWait;
  static bool outOfRange[_num_markers];
  static int marker1=-1, marker2=-1;
  static double marker1angle, marker2angle, marker1distance, marker2distance;

  if (state->outputs[HeadOffset+PanOffset]>M_PI/3
      && scanDirection==DIR_LEFT) {
    scanDirection=DIR_RIGHT;
    numDirSwitches++;
  }

  if (state->outputs[HeadOffset+PanOffset]<-M_PI/3
      && scanDirection==DIR_RIGHT) {
    scanDirection=DIR_LEFT;
    numDirSwitches++;
  }

  if (numDirSwitches>=2) {
    numDirSwitches=0;
    printf("localization failed\n");
    _locState=loc_disabled;
  }

  switch (_locState) {

    case loc_disabled:
      return;

    case loc_enabled:
      for (int i=0; i<_num_markers;i++) 
        outOfRange[i]=false;
      marker1=-1;
      if (_numMarkers>0) {
        _locState=loc_center1;
        marker1=_markers[markerClosestToCenter()].marker;
      } else {
        _locState=loc_moveto1;
      }
      break;

    case loc_moveto1:
      for (int i=0; i<_numMarkers; i++)
        if (!outOfRange[_markers[i].marker]) {
          marker1=_markers[i].marker;
          _locState=loc_center1;
          break;
        }
      if (marker1<0) moveHeadScreen(_centerX+scanDirection*50,_centerY);
      break;

    case loc_center1:
      numDirSwitches=0;
      if (abs_t(state->outputs[HeadOffset+PanOffset])>5*M_PI/12) {
        outOfRange[marker1]=true;
        marker1=-1;
        _locState=loc_moveto1;
        break;
      }
      if (_numMarkers>0) {
        int i;
        for (i=0; i<_numMarkers; i++)
          if (_markers[i].marker==marker1) break;
        if (_markers[i].marker!=marker1) { //dazed
          _locState=loc_moveto1;
          break;
        }
        if ((_markers[i].cen_x>_centerX-8) && (_markers[i].cen_x<_centerX+8)) {
          locWait=5;
          _locState=loc_wait1;
          marker2=-1;
          break;
        }
        moveHeadScreen(_markers[i].cen_x,_centerY);
      } else {
        _locState=loc_moveto1;
      }
      break;

    case loc_wait1:
      if (locWait<=0) {
        _locState=loc_moveto2;
        marker1angle=state->outputs[HeadOffset+PanOffset];
        marker1distance=state->sensors[IRDistOffset];
        if (marker1distance>=900.0) {
          outOfRange[marker1]=true;
          marker1=-1;
          _locState=loc_moveto1;
        }
      } else {
        locWait--; 
      }
      break;
      
    case loc_moveto2:
      for (int i=0; i<_numMarkers; i++)
        if (!outOfRange[_markers[i].marker] && _markers[i].marker!=marker1) {
          marker2=_markers[i].marker;
          _locState=loc_center2;
          break;
        }
      if (marker2<0) moveHeadScreen(_centerX+scanDirection*50,_centerY);
      break;

    case loc_center2:
      numDirSwitches=0;
      if (abs_t(state->outputs[HeadOffset+PanOffset])>5*M_PI/12) {
        outOfRange[marker2]=true;
        marker2=-1;
        _locState=loc_moveto2;
        break;
      }
      if (_numMarkers>0) {
        int i;
        for (i=0; i<_numMarkers; i++)
          if (_markers[i].marker==marker2) break;
        if (_markers[i].marker!=marker2) { //dazed
          _locState=loc_moveto2;
          break;
        }
        if ((_markers[i].cen_x>_centerX-8) && (_markers[i].cen_x<_centerX+8)) {
          locWait=5;
          _locState=loc_wait2;
          break;
        }
        moveHeadScreen(_markers[i].cen_x,_centerY);
      } else {
        _locState=loc_moveto2;
      }
      break;

    case loc_wait2:
      if (locWait<=0) {
        _locState=loc_done;
        marker2angle=state->outputs[HeadOffset+PanOffset];
        marker2distance=state->sensors[IRDistOffset];
        if (marker2distance>=900.0) {
          outOfRange[marker2]=true;
          marker2=-1;
          _locState=loc_moveto2;
        }
      } else {
        locWait--; 
      }
      break;
      
    case loc_done:
      _locState=loc_disabled;
      vision->disableEvents(MarkersSID);
      if (updatePosition(marker1, marker1distance, marker2, marker2distance)) {
        updateOrientation(marker2, marker2angle);
//	wmser->serializePos();
      }
      break;
      
    default:
      return;
  }
}

bool WorldModel::triangulate(int marker1, double distance1, int marker2, double distance2, double results[], int pos) {
  // x1, y1: world co-ordinates of marker1
  // x2, y2: world co-ordinates of marker2
  // d1, d2: distances of robot from respective markers

  double x1=_marker_x[marker1];
  double x2=_marker_x[marker2];
  double y1=_marker_y[marker1];
  double y2=_marker_y[marker2];
  double d1=distance1/10.0;
  double d2=distance2/10.0;
  double o, r, c, b, a, r1, r2, c1, c2;
 
  if (y2==y1) {
    // find o, r; s.t. x=ry+o
    o=(d1*d1-d2*d2+y2*y2-y1*y1+x2*x2-x1*x1)/(2*(x2-x1));
    r=(y1-y2)/(x2-x1);

    // find a, b, c; s.t. ay^2+by+c=0
    c=-d1*d1+y1*y1+x1*x1-2*x1*o+o*o;
    b=-2*x1*r+2*o*r-2*y1;
    a=1+r*r;

    // alright! solve for 2 values of y
    c1=(-b+sqrt(b*b-4*a*c))/(2*a);
    c2=(-b-sqrt(b*b-4*a*c))/(2*a);

    // calculate two values of x using x=ry+o
    r1=r*c1+o;
    r2=r*c2+o;
  } else {
    // find o, r; s.t. y=rx+o
    o=(d1*d1-d2*d2+y2*y2-y1*y1+x2*x2-x1*x1)/(2*(y2-y1));
    r=(x1-x2)/(y2-y1);

    // find a, b, c; s.t. ax^2+bx+c=0
    c=-d1*d1+y1*y1+x1*x1-2*y1*o+o*o;
    b=-2*y1*r+2*o*r-2*x1;
    a=1+r*r;
 
    // alright! solve for 2 values of x
    r1=(-b+sqrt(b*b-4*a*c))/(2*a);
    r2=(-b-sqrt(b*b-4*a*c))/(2*a);

    // calculate two values of y using y=rx+o
    c1=r*r1+o;
    c2=r*r2+o;
  }

  if (distance(0.0,0.0,r1,c1)<distance(0.0,0.0,r2,c2)
      && distance(0.0,0.0,r1,c1)<_poolRadius) {
    results[pos+0]=r1;
    results[pos+1]=c1;
  } else if (distance(0.0,0.0,r2,c2)<_poolRadius) {
    results[pos+0]=r2;
    results[pos+1]=c2;
  } else {
    return false;
  }
//  printf("got called with %d, %f, %d, %f. calculated %f, %f; %f, %f\n",marker1,distance1,marker2,distance2,r1,c1,r2,c2);
  return true;
}

bool WorldModel::updatePosition(int marker1, double distance1, int marker2, double distance2)
{
  double position[2];
  if (triangulate(marker1,distance1,marker2,distance2,position,0)) {
    _posX=position[0];
    _posY=position[1];
    _sdev=-1.0;
    return true;
  } else {
//    printf("localization failed\n");
    return false;
  }
}

bool WorldModel::updateOrientation(int marker, double markerangle)
{
  double x2=_marker_x[marker];
  double y2=_marker_y[marker];
  double headOrient=atan2(y2-_posY,x2-_posX);
  if (headOrient<0) headOrient+=2*M_PI;  // range from 0 to 2PI
  _bodyOrient=headOrient-markerangle;
  if (_bodyOrient<0) _bodyOrient+=2*M_PI;
  if (_bodyOrient>=2*M_PI) _bodyOrient-=2*M_PI;
  return true;
}

const char* WorldModel::markerToString(int mkr) {
  static const char* marker_strings[]=
   {"GOG","GOP","GPG","GPO","OGO","OGP","OPG","OPO","PGO","PGP","POG","POP"};
  return marker_strings[mkr];
}


int WorldModel::markerClosestToCenter() {
  int marker=0;
  int distance=squareDistance(_centerX,_centerY,
                              _markers[0].cen_x,_markers[0].cen_y);
  for (int i=1; i<_numMarkers; i++)
    if (squareDistance(_centerX,_centerY,
                       _markers[i].cen_x,_markers[i].cen_y)<distance)
      marker=i;
  return marker;
}

void WorldModel::moveHeadRadians(double tilt, double pan, double roll) {
  MMAccessor<HeadPointerMC>(headpointer_id).mc()->setJoints(
                     state->outputs[HeadOffset+TiltOffset]-tilt,
                     state->outputs[HeadOffset+PanOffset]-pan,
                     state->outputs[HeadOffset+RollOffset]-roll);
}

void WorldModel::moveHeadAbsRadians(double tilt, double pan, double roll) {
  MMAccessor<HeadPointerMC>(headpointer_id).mc()->setJoints(tilt,pan,roll);
}

void WorldModel::moveHead(int tilt, int pan, int roll) {
  moveHeadRadians(tilt/100.0*3.1415/5.5,
                  pan/100.0*3.1415/5.5,
                  roll/100.0*3.1415/5.5);
}

void WorldModel::moveHeadScreen(int x, int y) {
  moveHead((y-_centerY)*100/_centerY,(x-_centerX)*100/_centerX,0);
}

void WorldModel::removeObject(unsigned int obj) {
  if (_objects[obj]._lastseen!=0) {
    // delete from grid  
    int xpos=(int)(_objects[obj]._x/5+15.0);
    int ypos=(int)(_objects[obj]._y/5+15.0);
    _cells[xpos][ypos]=2;
  }
  _objects[obj]._lastseen=0;

  wmser->serializeGrid();
}

void WorldModel::reportObject(unsigned int obj) {
  if (_objects[obj]._lastseen!=0) {
    // delete from grid  
    int xpos=(int)(_objects[obj]._x/5+15.0);
    int ypos=(int)(_objects[obj]._y/5+15.0);
    _cells[xpos][ypos]=2;
  } else {
    _objects[obj]._lastseen=1;
  }
  double th=_bodyOrient+state->outputs[HeadOffset+PanOffset];
  double dist=state->sensors[IRDistOffset];

  _objects[obj]._x=_posX+cos(th)*dist/10;
  _objects[obj]._y=_posY+sin(th)*dist/10;

  int xpos=(int)(_objects[obj]._x/5+15.0);
  int ypos=(int)(_objects[obj]._y/5+15.0);
  cout << _posX << " " << _posY << " " << _bodyOrient << endl;
  cout << obj << " "  << xpos << " " << ypos << " " << th << " " << dist << endl;
  _cells[xpos][ypos]=(char)obj+4;

  wmser->serializeGrid();
}
