#ifndef __TIME_ET_CLASS__
#define __TIME_ET_CLASS__

#include <iostream>
#ifndef PLATFORM_APERIOS
#include <sys/time.h>
#else
#include <MCOOP.h>
//!would be defined by system
struct timeval {
	unsigned int tv_sec; //!< seconds
	long tv_usec;        //!< microseconds
};
//!would be defined by system
struct timezone {
	int     tz_minuteswest; //!< minutes west of Greenwich
	int     tz_dsttime;     //!< type of dst correction
};
#endif

//!a nice class for handling time values with high precision
/*@test negative times might not be handled properly */
class TimeET {
	//! lets the class be displayed easily
  friend std::ostream& operator<<(std::ostream& o, const TimeET& t);
 public:
	//@{
	//!constructor
  TimeET() : tv() {
    Set();
  }
  TimeET(long ms) : tv() {
    Set(ms);
  }
  TimeET(long sec, long usec) : tv() {
    Set(sec,usec);
  }
	//!constructor, sepecify @a t seconds
  TimeET(double t) :tv() {
    Set(t);
  }
	//@}

	//!returns the difference between the current time and the time stored
  inline TimeET Age() const { return TimeET()-(*this); }
	//@{
	//!returns the time stored as seconds in a double
  inline double Value() const { return (double)tv.tv_sec+(double)tv.tv_usec/(double)us_per_sec; }
	//@}
	
	//@{
	//!sets the time stored in the class
  inline void Set(long ms) {
    Set(0,ms*us_per_ms);
  }
  inline void Set(long sec, long usec) {
    tv.tv_sec=sec+usec/us_per_sec;
    tv.tv_usec=usec%us_per_sec;;
  }
  inline void Set(double t) {
    tv.tv_sec=(long)t;
    tv.tv_usec=(long)((t-tv.tv_sec)*us_per_sec);
  }
	/*!@brief sets the time to the current time
	 * @todo not getting timeofday on OPEN-R, is time since boot instead...*/
  inline void Set() {
#ifndef PLATFORM_APERIOS
    gettimeofday(&tv,&tz);
#else
		static struct SystemTime t;
		GetSystemTime(&t);
		Set(t.seconds,t.useconds);
#endif
  }
	//@}

	//@{
	//!for comparing times
  inline bool operator<(long ms) { //what if ms is negative?
    unsigned int sec = ms/ms_per_sec;
    return tv.tv_sec<sec || tv.tv_sec==sec && tv.tv_usec<static_cast<long>((ms-sec*ms_per_sec)*us_per_ms);
  }
  inline bool operator<(double t) {
    return Value()<t;
  }
  inline bool operator<(const TimeET& t) {
    return tv.tv_sec<t.tv.tv_sec || tv.tv_sec==t.tv.tv_sec && tv.tv_usec<t.tv.tv_usec;
  }
	//@}

	//@{
	//!for doing doing math with time
  inline TimeET operator+(const TimeET& t) const {
    long usec = tv.tv_usec+t.tv.tv_usec;
    long sec = tv.tv_sec+t.tv.tv_sec+usec/us_per_sec;
    usec%=us_per_sec;
    return TimeET(sec,usec);
  }
  inline TimeET operator+=(const TimeET& t) {
    tv.tv_usec+=t.tv.tv_usec;
    tv.tv_sec+=t.tv.tv_sec+tv.tv_usec/us_per_sec;
    tv.tv_usec%=us_per_sec;
    return *this;
  }
  inline TimeET operator-(const TimeET& t) const {
    long usec = tv.tv_usec-t.tv.tv_usec;
    long sec = tv.tv_sec-t.tv.tv_sec+usec/us_per_sec;
    usec%=us_per_sec;
    if(usec<0) {
      usec+=us_per_sec;
      sec--;
    }
    return TimeET(sec,usec);
  }
  inline TimeET operator-=(const TimeET& t) {
    tv.tv_usec-=t.tv.tv_usec;
    tv.tv_sec=tv.tv_sec-t.tv.tv_sec+tv.tv_usec/us_per_sec;
    tv.tv_usec%=us_per_sec;
    if(tv.tv_usec<0) {
      tv.tv_usec+=us_per_sec;
      tv.tv_sec--;
    }
    return *this;
  }
	//@}

  static const long us_per_sec=1000000; //!< conversion factor for microseconds to seconds
  static const long ms_per_sec=1000;    //!< conversion factor for milliseconds to seconds
  static const long us_per_ms=1000;     //!< conversion factor for microseconds to milliseconds
 protected:
  timeval tv; //!< stores the time
  static struct timezone tz; //!< stores the timezone (not really used)
};

//@{
//!for doing doing math with time
inline TimeET operator+(long t1, const TimeET& t2) { return TimeET(t1)+t2; }
inline TimeET operator-(double t1, const TimeET& t2) { return TimeET(t1)-t2; }
//@}

//! displays the value as text: secs~usecs
inline std::ostream& operator<<(std::ostream& o, const TimeET& t) {
  return (o << t.tv.tv_sec << '~' << t.tv.tv_usec);
}

/*! @file
 * @brief Describes TimeET, a nice class for handling time values with high precision
 * @author ejt (Creator)
 *
 * $Author: alokl $
 * $Name:  $
 * $Revision: 1.1.1.1 $
 * $State: Exp $
 * $Date: 2002/09/30 18:19:48 $
 */

#endif
