//-*-c++-*-
#ifndef INCLUDED_RobotInfo_h
#define INCLUDED_RobotInfo_h

#include <math.h>
#ifndef PLATFORM_APERIOS
typedef unsigned short word; //!< otherwise defined in Types.h
#else
#include <Types.h>
#endif

// Think you're l33t?  Figure out how to auto-sense these values so this can run on other models (220, 31x) without recompiling

//! Contains information about the Robot, such as number of joints, PID defaults, timing information, etc.
namespace RobotInfo {

	// *******************************
	//       ROBOT CONFIGURATION
	// *******************************


	const unsigned int FrameTime=8U;      //!< time between frames in the motion system (milliseconds)
	const unsigned int NumFrames=4;       //!< the number of frames per buffer (don't forget also double buffered)
	const unsigned int SlowFrameTime=128U; //!< time between frames for the ears (which move slower for some reason, don't want to mix with other outputs) (milliseconds)
	const unsigned int NumSlowFrames=1;    //!< the number of frames per buffer being sent to ears (double buffered as well)
	
	//!Corresponds to entries in PrimitiveName, defined at the end of this file, these are the primary grouping
	/*!Right now all binary joints are slow, but perhaps this won't always be the case... hence the IsFast/Slow bitmasks to select which type, in order to be more general */
	//!@name Output Types Information
	const unsigned NumPIDJoints   = 18; //!< The number of joints which use PID motion - everything except ears
	const unsigned NumLEDs        =  9; //!< The number LEDs which can be controlled
	const unsigned NumBinJoints   =  2; //!< The number of binary joints - just the ears
	const unsigned NumOutputs     = NumPIDJoints + NumBinJoints + NumLEDs; //!< the total number of outputs

	const bool IsFastOutput[NumOutputs] = { true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,true,false,false }; //!< true for joints which can be updated every 32 ms (all but the ears)
	const bool IsSlowOutput[NumOutputs] = { false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,false,true,true }; //!< true for joints which can only be updated every 128 ms (the ears)
	//@}

	const unsigned JointsPerLeg   =  3; //!< The number of joints per leg
	const unsigned NumLegs        =  4; //!< The number of legs
	const unsigned NumLegJoints   =  JointsPerLeg*NumLegs; //!< the TOTAL number of joints on ALL legs
	const unsigned NumHeadJoints  =  3; //!< The number of joints in the neck
	const unsigned NumTailJoints  =  2; //!< The number of joints assigned to the tail
	const unsigned NumMouthJoints =  1; //!< the number of joints that control the mouth
	const unsigned NumEarJoints   =  2; //!< The number of joints which control the ears (NOT per ear, is total)
	const unsigned NumButtons     =  8; //!< the number of buttons that are available, see ButtonOffset_t
	const unsigned NumSensors     =  1+3+1+5;  //!< 1 dist, 3 accel, 1 thermo, 5 from power, see SensorOffset_t
	

	// *******************************
	//         OUTPUT OFFSETS
	// *******************************


	//!Corresponds to entries in PrimitiveName, defined at the end of this file
	//!@name Output Offsets

	const unsigned PIDJointOffset = 0; //!< The beginning of the PID Joints
	const unsigned LegOffset   = PIDJointOffset;           //!< the offset of the beginning of the leg joints
	const unsigned HeadOffset  = LegOffset+NumLegJoints;   //!< the offset of the beginning of the head joints
	const unsigned TailOffset  = HeadOffset+NumHeadJoints; //!< the offset of the beginning of the tail joints
	const unsigned MouthOffset = TailOffset+NumTailJoints; //!< the offset of the beginning of the mouth joint

	const unsigned LEDOffset   = PIDJointOffset + NumPIDJoints; //!< the offset of LEDs in WorldState::outputs and MotionCommand functions

	const unsigned BinJointOffset = LEDOffset + NumLEDs; //!< The beginning of the binary joints
	const unsigned EarOffset   = BinJointOffset;           //!< the offset of the beginning of the ear joints - note that ears aren't sensed.  They can be flicked by the environment and you won't know.  Nor will they be flicked back
	//@}
	
	//! the ordering of legs
	enum LegOrder_t {
		LFrLegOrder = 0, //!< left front leg
		RFrLegOrder,     //!< right front leg
		LBkLegOrder,     //!< left back leg
		RBkLegOrder      //!< right back leg
	};
	
	//! The offsets of the individual legs
	enum LegOffset_t {
		LFrLegOffset = LegOffset+LFrLegOrder*JointsPerLeg, //!< beginning of left front leg
		RFrLegOffset = LegOffset+RFrLegOrder*JointsPerLeg, //!< beginning of right front leg
		LBkLegOffset = LegOffset+LBkLegOrder*JointsPerLeg, //!< beginning of left back leg
		RBkLegOffset = LegOffset+RBkLegOrder*JointsPerLeg  //!< beginning of right back leg
	};
	
	//! The offsets within appendages (the legs)  Note that the ordering matches the actual physical ordering of joints on the appendage (and not that of the head's TPROffset_t's)
	enum REKOffset_t {
		RotatorOffset=0, //!< moves leg forward or backward along body
		ElevatorOffset,  //!< moves leg toward or away from body
		KneeOffset       //!< moves knee
	};
	
	//! The offsets of appendages with tilt (elevation), pan (heading), and roll joints (i.e. head)  Note that the ordering matches the actual physical ordering of joints on the appendage (and not that of the leg's REKOffset_t's)
	enum TPROffset_t {
		TiltOffset = 0, //!< tilt/elevation (vertical)
		PanOffset,      //!< pan/heading (horizontal)
		RollOffset      //!< roll (rotational)
	};
	
	//! The offsets of the individual LEDs on the head and tail.  Note that L/R are robot's POV.  See also LEDBitMask_t
	enum LEDOffset_t {
		BotLLEDOffset = LEDOffset,//!< bottom left (red - sad)
		BotRLEDOffset, //!< bottom right (red - sad)
		MidLLEDOffset, //!< middle left (green - happy)
		MidRLEDOffset, //!< middle right (green - happy)
		TopLLEDOffset, //!< top left (red - angry)
		TopRLEDOffset, //!< top right (red - angry)
		TopBrLEDOffset,//!< top bar (green)
		TlRedLEDOffset,//!< red tail light
		TlBluLEDOffset //!< blue tail light
	};
	
	//! Bitmasks for use when specifying combinations of LEDs (see LEDEngine ) Note that L/R are robot's POV
	//!@name LED Bitmasks
	typedef unsigned int LEDBitMask_t; //!< So you can be clear when you're refering to a LED bitmask
	const LEDBitMask_t BotLLEDMask = 1<<(BotLLEDOffset-LEDOffset); //!< bottom left (red - sad)
	const LEDBitMask_t BotRLEDMask = 1<<(BotRLEDOffset-LEDOffset); //!< bottom right (red - sad)
	const LEDBitMask_t MidLLEDMask = 1<<(MidLLEDOffset-LEDOffset); //!< middle left (green - happy)
	const LEDBitMask_t MidRLEDMask = 1<<(MidRLEDOffset-LEDOffset); //!< middle right (green - happy)
	const LEDBitMask_t TopLLEDMask = 1<<(TopLLEDOffset-LEDOffset); //!< top left (red - angry)
	const LEDBitMask_t TopRLEDMask = 1<<(TopRLEDOffset-LEDOffset); //!< top right (red - angry)
	const LEDBitMask_t TopBrLEDMask= 1<<(TopBrLEDOffset-LEDOffset); //!< top bar (green)
	const LEDBitMask_t TlRedLEDMask= 1<<(TlRedLEDOffset-LEDOffset); //!< red tail light
	const LEDBitMask_t TlBluLEDMask= 1<<(TlBluLEDOffset-LEDOffset); //!< blue tail light
	const LEDBitMask_t FaceLEDMask = BotLLEDMask+BotRLEDMask+MidLLEDMask+MidRLEDMask+TopLLEDMask+TopRLEDMask+TopBrLEDMask; //!< LEDs for face (all but tail
	//@}


	// *******************************
	//          INPUT OFFSETS
	// *******************************


	//! The order in which inputs should be stored
	//!@name Input Offsets

	//! holds offsets to different buttons in WorldState::buttons[]
	/*! Should be a straight mapping to the ButtonSourceIDs
	 *
	 *  Note that the chest (power) button is not a normal button.  It kills
	 *  power to the motors at a hardware level, and isn't sensed in the
	 *  normal way.  If you want to know when it is pressed (and you are
	 *  about to shut down) see PowerSourceID::PauseSID.
	 *
	 *  @see WorldState::buttons @see ButtonSourceID_t */
	enum ButtonOffset_t {
		LFrPawOffset = LFrLegOrder,
		RFrPawOffset = RFrLegOrder,
		LBkPawOffset = LBkLegOrder,
		RBkPawOffset = RBkLegOrder,
		ChinButOffset= 4,
		BackButOffset,
		HeadFrButOffset, //!< not in reliable pressure units, but 1.0 is fairly stiff pressure, 0 is none
		HeadBkButOffset //!< not in reliable pressure units, but 1.0 is fairly stiff pressure, 0 is none
	};

	//! holds offset to different sensor values in WorldState::sensors[]
	/*! @see WorldState::sensors[] */
	enum SensorOffset_t {
		IRDistOffset = 0,  //!< in millimeters
		BAccelOffset, //!< backward acceleration, in @f$m/s^2@f$, negative if sitting on butt (positive for faceplant)
		LAccelOffset, //!< acceleration to the robot's left, in @f$m/s^2@f$, negative if lying on robot's left side
		DAccelOffset, //!< downward acceleration, in @f$m/s^2@f$, negative if standing up... be careful about the signs on all of these...
		ThermoOffset, //!< in degrees Celcius
		PowerRemainOffset, //!< percentage, 0-1
		PowerThermoOffset, //!<  degrees Celcius
		PowerCapacityOffset, //!< milli-amp hours
		PowerVoltageOffset, //!< volts
		PowerCurrentOffset //!< milli-amp negative values (maybe positive while charging?)
	};

	//@}


	//! The length of the strings used for each of the outputs in outputNames (doesn't include null term)
	const unsigned outputNameLen = 9;
	//! A name of uniform length for referring to joints - handy for posture files, etc.
	const char* const outputNames[NumOutputs] = {
		"LFr:rotor",
		"LFr:elvtr",
		"LFr:knee~",
		"RFr:rotor",
		"RFr:elvtr",
		"RFr:knee~",
		"LBk:rotor",
		"LBk:elvtr",
		"LBk:knee~",
		"RBk:rotor",
		"RBk:elvtr",
		"RBk:knee~",
		
		"NECK:tilt",
		"NECK:pan~",
		"NECK:roll",
		
		"TAIL:tilt",
		"TAIL:pan~",
		
		"MOUTH~~~~",
		
		"LED:botL~",
		"LED:botR~",
		"LED:midL~",
		"LED:midR~",
		"LED:topL~",
		"LED:topR~",
		"LED:topBr",
		
		"LED:tlRed",
		"LED:tlBlu",
		
		"EAR:left~",
		"EAR:right"
	};
	

	//! the joint identifier strings used to refer to specific joints in OPEN-R (but not needed for others)
	/*!@showinitializer 
	 * @warning IMPORTANT!!!!  DO NOT CHANGE THE ORDER OF ITEMS IN THIS TABLE!!!\n
	 *
	 * The offset consts defined in this file correspond to this table and will make life easier
	 * if you feel the need to reorder things, but they aren't used perfect @e everywhere \n
	 * In particular, assumptions are made that the pid joints will be in slots 0-numPIDJoints
	 * and that the fast outputs (ie NOT ears) will be in slots 0-NumFastOutputs\n
	 * There may be other assumptions not noted here!!!
	 * @note These entries DON'T correspond to the CPC index numbers defined in WorldState (this only lists joints, and in a different order defined by OPEN-R, that one has sensors as well*/
	const char* const PrimitiveName [NumOutputs] = {
		"PRM:/r2/c1-Joint2:j1",       //!< the left front leg   the rotator
		"PRM:/r2/c1/c2-Joint2:j2",    //!< the left front leg   the elevator 
		"PRM:/r2/c1/c2/c3-Joint2:j3", //!< the left front leg   the knee 
		"PRM:/r4/c1-Joint2:j1",       //!< the right front leg   the rotator
		"PRM:/r4/c1/c2-Joint2:j2",    //!< the right front leg    the elevator 
		"PRM:/r4/c1/c2/c3-Joint2:j3", //!< the right front leg   the knee 
		
		"PRM:/r3/c1-Joint2:j1",       //!< the left hind leg   the rotator
		"PRM:/r3/c1/c2-Joint2:j2",    //!< the left hind leg   the elevator 
		"PRM:/r3/c1/c2/c3-Joint2:j3", //!< the left hind leg   the knee
		"PRM:/r5/c1-Joint2:j1",       //!< the right hind leg   the rotator
		"PRM:/r5/c1/c2-Joint2:j2",    //!< the right hind leg   the elevator 
		"PRM:/r5/c1/c2/c3-Joint2:j3", //!< the right hind leg   the knee 

		"PRM:/r1/c1-Joint2:j1",       //!< the neck  tilt (12)
		"PRM:/r1/c1/c2-Joint2:j2",    //!< the neck   pan 
		"PRM:/r1/c1/c2/c3-Joint2:j3", //!< the neck   roll 
		
		"PRM:/r6/c2-Joint2:j2",       //!< the tail tilt (15)       // *** NOTE ***
		"PRM:/r6/c1-Joint2:j1",       //!< the tail pan             // CHANGE_ET 12/16/01 matches neck order
		
		"PRM:/r1/c1/c2/c3/c4-Joint2:j4", //!< the mouth (17)
		
		"PRM:/r1/c1/c2/c3/l1-LED2:l1", //!< lower  left  LED (18)
		"PRM:/r1/c1/c2/c3/l4-LED2:l4", //!< lower  right LED
		"PRM:/r1/c1/c2/c3/l2-LED2:l2", //!< middle left  LED
		"PRM:/r1/c1/c2/c3/l5-LED2:l5", //!< middle right LED
		"PRM:/r1/c1/c2/c3/l3-LED2:l3", //!< upper  left  LED
		"PRM:/r1/c1/c2/c3/l6-LED2:l6", //!< upper  right LED
		"PRM:/r1/c1/c2/c3/l7-LED2:l7", //!< top          LED
		
		"PRM:/r6/l2-LED2:l2", //!< tail red  LED
		"PRM:/r6/l1-LED2:l1", //!< tail blue LED
		
		"PRM:/r1/c1/c2/c3/e1-Joint3:j5", //!< left ear (27)
		"PRM:/r1/c1/c2/c3/e2-Joint3:j6" //!< right ear
	};

	/*const word Pid[NumPIDJoints][6] =
		{
			{ 0x16, 0x04, 0x08, 0x0E, 0x02, 0x0F },
			{ 0x14, 0x04, 0x06, 0x0E, 0x02, 0x0F },
			{ 0x23, 0x04, 0x05, 0x0E, 0x02, 0x0F },
			{ 0x16, 0x04, 0x08, 0x0E, 0x02, 0x0F },
			{ 0x14, 0x04, 0x06, 0x0E, 0x02, 0x0F },
			{ 0x23, 0x04, 0x05, 0x0E, 0x02, 0x0F },
			{ 0x16, 0x04, 0x08, 0x0E, 0x02, 0x0F },
			{ 0x14, 0x04, 0x06, 0x0E, 0x02, 0x0F },
			{ 0x23, 0x04, 0x05, 0x0E, 0x02, 0x0F },
			{ 0x16, 0x04, 0x08, 0x0E, 0x02, 0x0F },
			{ 0x14, 0x04, 0x06, 0x0E, 0x02, 0x0F },
			{ 0x23, 0x04, 0x05, 0x0E, 0x02, 0x0F },
			
			{ 0x0A, 0x08, 0x0C, 0x0E, 0x02, 0x0F },
			{ 0x0D, 0x08, 0x0B, 0x0E, 0x02, 0x0F },
			{ 0x10, 0x08, 0x0C, 0x0E, 0x02, 0x0F }, // P was 0x0C, updated as seen on https://www.openr.org/page1_2001/gain.html 8/13/2002
			
			{ 0x0A, 0x00, 0x18, 0x0E, 0x02, 0x0F },
			{ 0x07, 0x00, 0x11, 0x0E, 0x02, 0x0F },
			
			{ 0x0E, 0x08, 0x10, 0x0E, 0x02, 0x0F }, //  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
			};*/
	//! This table holds the default PID values for each joint.  @see PIDMC
	const float DefaultPIDs[NumPIDJoints][3] =
		{
			{ 0x16/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x08/(double)(1<<(16-0xF)) },
			{ 0x14/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x06/(double)(1<<(16-0xF)) },
			{ 0x23/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x05/(double)(1<<(16-0xF)) },
			{ 0x16/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x08/(double)(1<<(16-0xF)) },
			{ 0x14/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x06/(double)(1<<(16-0xF)) },
			{ 0x23/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x05/(double)(1<<(16-0xF)) },
			{ 0x16/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x08/(double)(1<<(16-0xF)) },
			{ 0x14/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x06/(double)(1<<(16-0xF)) },
			{ 0x23/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x05/(double)(1<<(16-0xF)) },
			{ 0x16/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x08/(double)(1<<(16-0xF)) },
			{ 0x14/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x06/(double)(1<<(16-0xF)) },
			{ 0x23/(double)(1<<(16-0xE)), 0x04/(double)(1<<(16-0x2)), 0x05/(double)(1<<(16-0xF)) },

			{ 0x0A/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x0C/(double)(1<<(16-0xF)) },
			{ 0x0D/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x0B/(double)(1<<(16-0xF)) },
			{ 0x10/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x0C/(double)(1<<(16-0xF)) },

			{ 0x0A/(double)(1<<(16-0xE)), 0x00/(double)(1<<(16-0x2)), 0x18/(double)(1<<(16-0xF)) },
			{ 0x07/(double)(1<<(16-0xE)), 0x00/(double)(1<<(16-0x2)), 0x11/(double)(1<<(16-0xF)) },

			{ 0x0E/(double)(1<<(16-0xE)), 0x08/(double)(1<<(16-0x2)), 0x10/(double)(1<<(16-0xF)) }
		};
	
#ifndef RAD
	//!Just a little macro for converting degrees to radians
#define RAD(deg) (((deg) * M_PI ) / 180.0)
	//!a flag so we undef these after we're done - do you have a cleaner solution?
#define __RI_RAD_FLAG
#endif

	//! Defines the min and max index of entries in #outputRanges and #mechanicalLimits
	enum MinMaxRange_t { MinRange,MaxRange };

	//! This table holds the mechanical limits of each of the outputs
	const double outputRanges[NumOutputs][2] =
		{
			{ RAD(-117),RAD(117) },{ RAD(-11),RAD(89) },{ RAD(-27),RAD(147) }, //left front REK
			{ RAD(-117),RAD(117) },{ RAD(-11),RAD(89) },{ RAD(-27),RAD(147) }, //right front REK
			{ RAD(-117),RAD(117) },{ RAD(-11),RAD(89) },{ RAD(-27),RAD(147) }, //left back REK
			{ RAD(-117),RAD(117) },{ RAD(-11),RAD(89) },{ RAD(-27),RAD(147) }, //right back REK

			{ RAD(-82),RAD(43) },{ RAD(-89.6),RAD(89.6) },{ RAD(-29),RAD(29) }, //neck TPR
				
			{ RAD(-22),RAD(22) },{ RAD(-22),RAD(22) }, // tail tp

			{ RAD(-47),RAD(-3) }, //mouth

			{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1}, //LEDs
		
			{0,1},{0,1} //ears
		};

	//! This table holds the mechanical limits of each of the outputs
	const double mechanicalLimits[NumOutputs][2] =
		{
			{ RAD(-120),RAD(120) },{ RAD(-14),RAD(92) },{ RAD(-30),RAD(150) }, //left front jsk
			{ RAD(-120),RAD(120) },{ RAD(-14),RAD(92) },{ RAD(-30),RAD(150) }, //right front jsk
			{ RAD(-120),RAD(120) },{ RAD(-14),RAD(92) },{ RAD(-30),RAD(150) }, //left back jsk
			{ RAD(-120),RAD(120) },{ RAD(-14),RAD(92) },{ RAD(-30),RAD(150) }, //right back jsk

			{ RAD(-85),RAD(46) },{ RAD(-92.6),RAD(92.6) },{ RAD(-32),RAD(32) }, //neck tpr
				
			{ RAD(-25),RAD(25) },{ RAD(-25),RAD(25) }, // tail tp

			{ RAD(-50),RAD(0) }, //mouth

			{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1},{0,1}, //LEDs
		
			{0,1},{0,1} //ears
		};

#ifdef __RI_RAD_FLAG
#undef RAD
#undef __RI_RAD_FLAG
#endif

}

using namespace RobotInfo;
	
/*! @file
 * @brief Defines in part RobotInfo namespace, gives some information about the robot's capabilities, such as joint counts, offsets, names and PID values
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name:  $
 * $Revision: 1.7 $
 * $State: Exp $
 * $Date: 2003/02/23 09:58:15 $
 */









// This information is also documented in MotionManager::setPID(unsigned int joint, const word pids[6])

/*
> Date: Tue, 29 May 2001 22:09:03 +0900
> From: openr-robocup <openr-robocup-admin@pdp.crl.sony.co.jp>
> Subject: [openr-robocup-info(Sony Confidential):00038] Re: Fwd: data about new
>  robot motor joints
> To: Vincent HUGEL <hugel@robot.uvsq.fr>
> Cc: openr-robocup-info@pdp.crl.sony.co.jp
>  
> Dear Vincent,
>  
> The data on the motor joints of ERS-2100 is as below.
>  
> - Maximum output torque at each actuator
>  
> Head : Tilt  2.4kgfcm
>           Pan    2kgfcm
>           Roll    1kgfcm
> Leg : Joint   5kgfcm
>          Shoulder  4kgfcm
>          Knee  4.3kgfcm
>  
> - Idling speed
>  
> Head : Tilt   51rpm
>           Pan  101rpm
>           Roll   127rpm
> Leg : Joint  51rpm
>          Shoulder  55rpm
>          Knee  55rpm
>  
*/


/*
> Pid: pgain, igain, dgain, pshift, ishift, dshift
> 
> PGAIN: Proportional gain (16bit depth)
> IGAIN: Integral gain (16bit depth)
> DGAIN: Differential gain (16bit depth)
>  
> PSHIFT: Proportional gain shift amount (4bit)
> ISHIFT: Proportional gain shift amount  (4bit)
> DSHIFT: Proportional gain shift amount  (4bit)
>  
> Operation :
>  
> P, I, D, value, respectively
> Multiple the gain G to the error between the desired value Vd and current
> value Vc,
>      F = (Vd - Vc ) * G
> and shift (0x10 - SHIFT) bit right side.
>      F >> SHIFT
> Namely, SHIFT=0x0E  (0x10 - 0x0E) = 2 bit shift
> Assume Vd = 50, Vc= 20, Gain = 0x16=22, SHIFT=0xE,
> The operation will be
>      (50 - 20) * 22 = 660 =1010010100
>      1010010100 >> 2 = 10100101 =  165
*/

//const int lp = 40;
//const int li = 2;
//
//const int p_gain[numOfJoint] = { lp, lp, lp, lp, lp, lp,
//				   lp, lp, lp, lp, lp, lp,
//				   12,  8, 12, // 12 12 12
//				   16, 16, 16};
//
//const int i_gain[numOfJoint] = { li, li, li, li, li, li,
//				   10, li, li, 10, li, li,
//				   18,  8, 24, // 64 10 64
//				   5,  5,  5};
//
//const int d_gain[numOfJoint] = { 50, 50, 50, 50, 50, 50,
//				   50, 50, 50, 50, 50, 50,
//				   24,  8, 24, // 64 10 64
//				   5,  5,  5};

#endif
