//-*-c++-*-
#ifndef INCLUDED_SharedMotion_h
#define INCLUDED_SharedMotion_h

#include "MotionManagerMsg.h"
#include <OPENR/RCRegion.h>

//! You need a subclass of SharedMotionBase so that a MotionCommand can be wrapped in a shared memory region for passing to OPEN-R, SharedMotion is what you actually want to read about
/*! See MotionManager for an example on how to use this. @see MotionManager @see SharedMotion<MC> */
class SharedMotionBase {
public:
	RCRegion * getRegion() const { return rcr; } //!< returns the OPEN-R memory region
	MotionManagerMsg* MotionManagerMsgBase() const { return (MotionManagerMsg*)(rcr->Base()); } //!< returns a pointer to a structure to designate what is in the shared memory region to receivers
	MotionCommand* DataBase() const { return reinterpret_cast<MotionCommand*>(rcr->Base()+MotionManagerMsg::SIZEOF_MSG); } //!< returns a pointer to the data region (following the MotionManagerMsg)

protected:
	SharedMotionBase() : rcr(NULL) {} //!< constructor, protected because you shouldn't need to create this directly, just a common interface to all templates of SharedMotion
	RCRegion * rcr; //!< the pointer to the shared memory region this is in charge of
	
private:
	SharedMotionBase(const SharedMotionBase&); //!< this shouldn't be called...
	SharedMotionBase& operator=(const SharedMotionBase&); //!< this shouldn't be called...
};	

//! This templated class allows convenient creation of any type of MotionCommand subclass wrapped in a shared memory region
/*! See MotionManager for an example on how to use this. @see MotionManager @see SharedMotionBase */
template<class MC>
class SharedMotion : public SharedMotionBase {
public:
	//! Creates the MotionCommand with the default constructor
	SharedMotion() : SharedMotionBase() {
		rcr = new RCRegion(calcsize());
		new (rcr->Base()+MotionManagerMsg::SIZEOF_MSG) MC;
	}
	//! Creates the MotionCommand, passing its constructor t1
	template<class T1> SharedMotion(T1 t1) : SharedMotionBase() {
		rcr = new RCRegion(calcsize());
		new (rcr->Base()+MotionManagerMsg::SIZEOF_MSG) MC(t1);
	}
	//! Creates the MotionCommand, passing its constructor t1 and t2
	template<class T1, class T2> SharedMotion(T1 t1, T2 t2) : SharedMotionBase(){
		rcr = new RCRegion(sizeof(MotionManagerMsg)+sizeof(MC));
		new (rcr->Base()+sizeof(MotionManagerMsg)) MC(t1,t2);
	}
	//! Creates the MotionCommand, passing its constructor t1, t2, and t3
	template<class T1, class T2, class T3> SharedMotion(T1 t1, T2 t2, T3 t3) : SharedMotionBase(){
		rcr = new RCRegion(sizeof(MotionManagerMsg)+sizeof(MC));
		new (rcr->Base()+sizeof(MotionManagerMsg)) MC(t1,t2,t3);
	}
	//! Creates the MotionCommand, passing its constructor t1, t2, t3 and t4
	template<class T1, class T2, class T3, class T4> SharedMotion(T1 t1, T2 t2, T3 t3, T4 t4) : SharedMotionBase(){
		rcr = new RCRegion(sizeof(MotionManagerMsg)+sizeof(MC));
		new (rcr->Base()+sizeof(MotionManagerMsg)) MC(t1,t2,t3,t4);
	}
	//! Creates the MotionCommand, passing its constructor t1, t2, t3, t4 and t5 - if you need more arguments, just add them
	template<class T1, class T2, class T3, class T4, class T5> SharedMotion(T1 t1, T2 t2, T3 t3, T4 t4, T5 t5) : SharedMotionBase(){
		rcr = new RCRegion(sizeof(MotionManagerMsg)+sizeof(MC));
		new (rcr->Base()+sizeof(MotionManagerMsg)) MC(t1,t2,t3,t4,t5);
	}
	//if you really need more than 5 arguments for your MotionCommand, well, you're one crazy puppy
	//but if you really want to, just make more like above... (yay templates!)

	MC* MCBase() const { return (MC*)DataBase(); } //!< returns a correctly typed pointer to the MotionCommand's memory
	MC* operator->() const { return MCBase(); } //!< smart pointer to the underlying MotionCommand
	MC& operator*() const { return *MCBase(); } //!< smart pointer to the underlying MotionCommand
	MC& operator[](int i) const { return MCBase()[i]; } //!< smart pointer to the underlying MotionCommand
protected:
	//!Calculates the size of the memory region to be used, rounding up to the nearest page size
	unsigned int calcsize() {
		size_t size = MotionManagerMsg::SIZEOF_MSG+sizeof(MC);
		sError error;
		size_t page_size;
		error = GetPageSize(&page_size);
		if (error != sSUCCESS) {
			cout << "error: " << error << " getting page size in SharedMem" << endl;
			page_size = 4096;
		}
		
		int new_size,num_pages;
		num_pages = (size+page_size-1)/page_size;
		new_size = num_pages*page_size;
		//cout << "req" << size << "new" << new_size << "ps" << page_size << endl;
		/*		cout << "data size is " << sizeof(MC) << endl;
					cout << "msg size is " << MotionManagerMsg::SIZEOF_MSG << endl;
					cout << "SIZE is " << rcr->Size() << endl;
					cout << "PAGE is " << page_size << endl;
					cout << "BASE is " << (void*)rcr->Base() << endl; */
		return new_size;
	}
};

/*! @file
 * @brief Defines SharedMotion, a wrapper for MotionCommands in order to facilitate sending them between processes
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name:  $
 * $Revision: 1.2 $
 * $State: Exp $
 * $Date: 2003/01/09 02:02:59 $
 */

#endif //INCLUDED_SharedMotion_h
