#include "PostureEngine.h"
#include "Shared/WorldState.h"
#include <stdio.h>

PostureEngine::~PostureEngine() {}

void PostureEngine::takeSnapshot() {
	for(unsigned int i=0; i<NumOutputs; i++)
		cmds[i].set(state->outputs[i],1);
}

void PostureEngine::clear() {
	for(unsigned int i=0; i<NumOutputs; i++)
		cmds[i].unset();
}

PostureEngine& PostureEngine::setOverlay(const PostureEngine& pe) {
	for(unsigned int i=0; i<NumOutputs; i++)
		if(pe.cmds[i]!=unusedJoint)
			cmds[i]=pe.cmds[i];
	return *this;
}
PostureEngine PostureEngine::createOverlay(const PostureEngine& pe) const {
	PostureEngine tmp(*this);
	return tmp.setOverlay(pe);
}
PostureEngine& PostureEngine::setUnderlay(const PostureEngine& pe) {
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i]==unusedJoint)
			cmds[i]=pe.cmds[i];
	return *this;
}
PostureEngine PostureEngine::createUnderlay(const PostureEngine& pe) const {
	PostureEngine tmp(*this);
	return tmp.setUnderlay(pe);
}
/*! joints being averaged with ::unusedJoint have their weights averaged, but not their values (so an output can crossfade properly)\n
 *  @param pe the other PostureEngine
 *  @param w amount to weight towards @a pe
 *  - if @a w < .001, nothing is done
 *  - if @a w > .999, a straight copy of @a pe occurs (sets joints to ::unusedJoint properly at end of fade)
 *  - .001 and .999 is used instead of 0 and 1 to allow for slight addition errors in a loop (if
 *    using repeated additions of a delta value instead of repeated divisions)
 *  @return @c *this, stores results into this */
PostureEngine& PostureEngine::setAverage(const PostureEngine& pe, double w) {
	if(w<0.001)
		return *this;
	if(w>0.999)
		return (*this=pe);
	double wp=1-w;
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i]!=unusedJoint) {
			if(pe.cmds[i]!=unusedJoint)
				cmds[i].set(cmds[i].value*wp+pe.cmds[i].value*w,cmds[i].weight*wp+pe.cmds[i].weight*w);
			else
				cmds[i].weight*=wp;
		} else
			cmds[i].set(pe.cmds[i].value,pe.cmds[i].weight*w);
	return *this;
}
/*! joints being averaged with ::unusedJoint have their weights averaged, but not their values (so an output can crossfade properly)\n
 *  @param pe the other PostureEngine
 *  @param w amount to weight towards @a pe
 *  - if @a w < .001, nothing is done
 *  - if @a w > .999, a straight copy of @a pe occurs (sets joints to ::unusedJoint properly at end of fade)
 *  - .001 and .999 is used instead of 0 and 1 to allow for slight addition errors in a loop (if
 *    using repeated additions of a delta value instead of repeated divisions)
 *  @return a new posture containing the results */
PostureEngine PostureEngine::createAverage(const PostureEngine& pe, double w) const {
	PostureEngine tmp(*this);
	return tmp.setAverage(pe,w);
}
PostureEngine& PostureEngine::setCombine(const PostureEngine& pe) {
	for(unsigned int i=0; i<NumOutputs; i++) {
		double total=cmds[i].weight+pe.cmds[i].weight;
		cmds[i].set((cmds[i].value*cmds[i].weight+pe.cmds[i].value*pe.cmds[i].weight)/total,total);
	}
	return *this;
}
PostureEngine PostureEngine::createCombine(const PostureEngine& pe) const {
	PostureEngine tmp(*this);
	return tmp.setCombine(pe);
}

double PostureEngine::diff(const PostureEngine& pe) const {
	double ans=0;
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i]!=unusedJoint && pe.cmds[i]!=unusedJoint) {
			double dif=cmds[i].value-pe.cmds[i].value;
			ans+=dif*dif;
		}
	return ans;
}

double PostureEngine::avgdiff(const PostureEngine& pe) const {
	double ans=0;
	unsigned int cnt=0;
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i]!=unusedJoint && pe.cmds[i]!=unusedJoint) {
			double dif=cmds[i].value-pe.cmds[i].value;
			ans+=dif*dif;
			cnt++;
		}
	return ans/cnt;
}

double PostureEngine::maxdiff(const PostureEngine& pe) const {
	double max=0;
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i]!=unusedJoint && pe.cmds[i]!=unusedJoint) {
			double dif=cmds[i].value-pe.cmds[i].value;
			dif*=dif;
			if(dif>max)
				max=dif;
		}
	return max;
}

unsigned int PostureEngine::getBinSize() const {
	unsigned int len=10;
	char buf[255];
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i]!=unusedJoint)
			len+=snprintf(buf,len,"%s\t% .4f\t% .4f\n",outputNames[i],cmds[i].value,cmds[i].weight);
	return len;
}

unsigned int PostureEngine::LoadBuffer(const char buf[], unsigned int len) {
	unsigned int origlen=len;
	clear();
	if(strncmp("#POS\n",buf,5)!=0) {
		//		cout << "ERROR PostureEngine load corrupted - expected #POS header" << endl;
		return 0;
	}
	char formatstring[64];
	snprintf(formatstring,64,"%%%dc %%f %%f\n%%n",outputNameLen); //cout << "Format: " << formatstring << endl;
	unsigned int idx=0;
	unsigned int linenum=2;
	char jname[outputNameLen+1];
	jname[outputNameLen]='\0';
	while(len<=origlen && len>0) {
		float fval, fwht;
		int written;
		//		printf("%d %.9s\n",linenum+1,buf);
		if(buf[0]=='#' || buf[0]=='\n') {
			if(strncmp("#END\n",buf,5)==0)
				return origlen-len;
			else {
				while(*buf++!='\n') {}
				continue;
			}
		}
		written=-1;
		sscanf(buf,formatstring,jname,&fval,&fwht,&written);
		if(!ChkAdvance(written,&buf,&len,"*** ERROR PostureEngine load corrupted - line %d\n",linenum)) return 0;
		linenum++;
		//cout << '"' << jname << "\"\t" << (double)fval << '\t' << (double)fwht << endl;
		unsigned int startidx=idx+1;
		for(;idx<NumOutputs;idx++)
			if(strcmp(jname,outputNames[idx])==0) {
				cmds[idx].set(fval,fwht);
				break;
			}
		if(idx==NumOutputs) {
			for(idx=0;idx<startidx;idx++)
				if(strcmp(jname,outputNames[idx])==0) {
					cmds[idx].set(fval,fwht);
					break;
			}
			if(idx==startidx && strcmp(jname,outputNames[idx])!=0)
				cout << "*** WARNING unmatched jointName: " << jname << endl;
		}
	}
	cout << "*** WARNING PostureEngine load missing #END" << endl;
	return origlen-len;
}

//Why do i need a cast to const char**??? It doesn't work without, should be automatic... grrrr
unsigned int PostureEngine::SaveBuffer(char buf[], unsigned int len) const {
	unsigned int origlen=len;
	int written=snprintf(buf,len,"#POS\n");
	if(!ChkAdvance(written,(const char**)&buf,&len,"*** ERROR PostureEngine save failed on header\n")) return 0;
	if(len==0 || len>origlen) {
		cout << "*** ERROR PostureEngine save overflow on header" << endl;
		return 0;
	}
	for(unsigned int i=0; i<NumOutputs; i++)
		if(cmds[i]!=unusedJoint) {
			written=snprintf(buf,len,"%s\t% .4f\t% .4f\n",outputNames[i],cmds[i].value,cmds[i].weight);
			if(!ChkAdvance(written,(const char**)&buf,&len,"*** ERROR PostureEngine save failed\n")) return 0;
			if(len==0 || len>origlen) {
				cout << "*** ERROR PostureEngine save overflow" << endl;
				return 0;
			}
		}
	written=snprintf(buf,len,"#END\n");
	if(!ChkAdvance(written,(const char**)&buf,&len,"*** ERROR PostureEngine save failed on #END\n")) return 0;
	if(len==0 || len>origlen) {
		cout << "*** ERROR PostureEngine save overflow on #END" << endl;
		return 0;
	}
	return origlen-len;
}

bool PostureEngine::ChkAdvance(int res, const char** buf, unsigned int* len, const char* msg) {
	if(res>0) {
		*buf+=res;
		*len-=res;
		return true;
	} else {
		printf("%s",msg);
		return false;
	}
}

bool PostureEngine::ChkAdvance(int res, const char** buf, unsigned int* len, const char* msg, int arg1) {
	if(res>0) {
		*buf+=res;
		*len-=res;
		return true;
	} else {
		printf(msg,arg1);
		return false;
	}
}

/*! @file
 * @brief Implements PostureEngine, a base class for managing the values and weights of all the outputs
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name:  $
 * $Revision: 1.4 $
 * $State: Exp $
 * $Date: 2003/01/09 02:02:59 $
 */

