//-*-c++-*-
#ifndef INCLUDED_PIDMC_h_
#define INCLUDED_PIDMC_h_

#include "MotionCommand.h"

//! A nice little MotionCommand for manually manipulating the PID values
/*! This can be a one shot deal if you set the autoprune flag as you're adding it (which is the default) */
class PIDMC : public MotionCommand {
public:
	//!Constructor, uses default PIDs and 0 weight for all
	PIDMC() : MotionCommand(), dirty(false) {
		setDefaults(0);
		dirty=false;
	}
	//!Constructor, sets general power level of all 
	PIDMC(float powerlevel, float w=1) : MotionCommand(), dirty(false) {
		setAllPowerLevel(powerlevel,w);
	}
	//!Constructor, sets general power level of a range of joints, uses default and 0 weight for others 
	PIDMC(unsigned int low, unsigned int high, float powerlevel, float w=1) : MotionCommand(), dirty(false) {
		setRangePowerLevel(PIDJointOffset,low,1.f,0.f);
		setRangePowerLevel(low,high,powerlevel,w);
		setRangePowerLevel(high,PIDJointOffset+NumPIDJoints,1.f,0.f);
	}
	//!Destructor
	virtual ~PIDMC() {}

	//!Inherited
	//@{
	virtual int updateJointCmds() { int wasDirty=dirty; dirty=false; return wasDirty; }
	virtual const JointCmd& getJointCmd(unsigned int) { return unusedJoint; } /*!<doesn't use any joints*/
	virtual int isDirty() { return dirty; }
	virtual int isAlive() { return dirty; }
	virtual bool isUsingOutput(unsigned int i) { return isPID(i) && weights[i]>0; }
	virtual float* getPIDValue(unsigned int i, float& weight) {
		i-=PIDJointOffset;
		weight=weights[i];
		return PIDs[i];
	}
	//@}

	//!Sets the PIDs to the defaults specified in RobotInfo
	void setDefaults(float weight=1) {
		setAllPowerLevel(1.f,weight);
	}

	//!Sets the PIDs to a percentage of default for a given joint, and sets weight
	void setJointPowerLevel(unsigned int i, float p, float w=1) {
		i-=PIDJointOffset;
		for(unsigned int j=0;j<3;j++)
			PIDs[i][j]=DefaultPIDs[i][j]*p;
		weights[i]=w;
		dirty=true;
	}

	//!Sets the PIDs to a percentage of default for all joints
	void setAllPowerLevel(float p, float w=1) {
		for(unsigned int i=0; i<NumPIDJoints; i++) {
			for(unsigned int j=0;j<3;j++)
				PIDs[i][j]=DefaultPIDs[i][j]*p;
			weights[i]=w;
		}
		dirty=true;
	}

	//!Sets a range of joints' PIDs to a given power level and weight
	void setRangePowerLevel(unsigned int low, unsigned int high, float p, float w=1) {
		low-=PIDJointOffset;
		high-=PIDJointOffset;
		for(unsigned int i=low; i<high; i++) {
			for(unsigned int j=0;j<3;j++)
				PIDs[i][j]=DefaultPIDs[i][j]*p;
			weights[i]=w;
		}
		dirty=true;
	}


	//!Use this to set the PID value and weight
	void setPIDValue(unsigned int i, const float pid[3], float w=1) {
		i-=PIDJointOffset;
		for(unsigned int j=0;j<3;j++)
			PIDs[i][j]=pid[j];
		weights[i]=w;
		dirty=true;
	}

	//!Use this if you want to double check the PID you set
	const float* getPIDValue(unsigned int i) const {
		return PIDs[i-PIDJointOffset];
	}

	//!Use this to set the weight used in averaging PID values (if more than one MC is setting them)
	void setWeight(unsigned int i, float w) {
		weights[i-PIDJointOffset]=w;
		dirty=true;
	}
	//!Use this to get the weight used in averaging PID values (if more than one MC is setting them)
	double getWeight(unsigned int i) const {
		if(isPID(i))
			return weights[i-PIDJointOffset];
		else
			return 0;
	}

protected:
	//! returns true if the output i is a PID joint
	static inline bool isPID(unsigned int i) {
		return ((int)i>=(int)PIDJointOffset && i<PIDJointOffset+NumPIDJoints); //casting to int just to get rid of compiler warning.. sigh
	}

	bool dirty; //!< true if there are changes that have not been picked up by Motion
	float weights[NumPIDJoints]; //!< the weights to use in requesting PID values
	float PIDs[NumPIDJoints][3]; //!< the PIDs being requested
};

/*! @file
 * @brief 
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name:  $
 * $Revision: 1.6 $
 * $State: Exp $
 * $Date: 2003/02/23 09:58:14 $
 */

#endif
