#include "MotionManager.h"
#include "Shared/debuget.h"
#include "Shared/WorldState.h"
#include <list>

MotionManager * motman=NULL;
unsigned int _MMaccID=-1U;

//! just for convenience
typedef unsigned int uint;

MotionManager::MotionManager()
	: pidchanges(),cmdlist(),MMlock(),numAcc(0)
{
	for(uint x=0; x<NumOutputs; x++)
		cmdSums[x]=0;
}

void
MotionManager::InitAccess(OSubject* subj) {
	if(numAcc==MAX_ACCESS) {
		printf("*** ERROR *** attempted to register more accessors with MotionManager than allowed by MAX_ACCESS\n");
		return;
	}
	_MMaccID=numAcc++;
 	//	cout << "ID is now " << _MMaccID << " of " << numAcc << endl;
	//	cout << "_MMaccID is " << &_MMaccID << endl;
	//	cout << "numAcc is " << &numAcc << " from " << this << endl;
	MMlock.lock(_MMaccID);
	//	accRegs[accID].init();
	subjs[_MMaccID]=subj;
	if(cmdlist.size()>0) //Shouldn't happen - busy wait in addMotion
		cout << "*** WARNING *** MOTIONS ADDED BEFORE ALL INITACCESSED" << endl;
	MMlock.release();
}

void
MotionManager::getOutputs(float outputs[NumFrames][NumOutputs]) {
	//	if(begin(id)!=end())
	//if(state && state->buttons[LFrPawOffset]) cout << "getAngles..." << flush;
	if(state==NULL) {
		// we haven't gotten the WorldState memory region from Main yet, just set LEDs to a wierd pattern and leave
		for(uint f=0;f<NumFrames;f++)
			for(uint i=0; i<NumOutputs; i++)
				outputs[f][i]=0;
		for(uint f=0;f<NumFrames;f++)
			for(uint l=0; l<NumLEDs; l++)
				outputs[f][l]=l/(NumLEDs-1.0);
		//	if(begin(id)!=end())
		//if(state && state->buttons[LFrPawOffset])	cout << "getangles-nostate-done..." << flush;
		return;
	}
	accID_t id = func_begin();
	//	if(begin(id)!=end())
	//	cout << id << "..." << flush;
	//	cout << "CHECKOUT..." << flush;
	for(uint output=0; output<NumOutputs; output++)
		cmdstates[output].clear();

	cleanup(id);
	
	list<MC_ID> unlocked;
	for(MC_ID it=begin(id); it!=end(); it=next(it,id)) // check out all the MotionCommands (only one at a time tho)
		unlocked.push_back(it);
	while(unlocked.size()>0) { // keep cycling through all the locks we didn't get
		for(list<MC_ID>::iterator it=unlocked.begin(); it!=unlocked.end(); ) {
			MotionCommand* mc=checkoutMotion(*it,false,id);
			if(mc==NULL)
				it++; //we didn't get a lock, skip it (we'll keep looping until we get it)
			else {
				// we got a lock
				mc->updateJointCmds();
				float p = mc->getPriority();
				if(p<MotionCommand::kBackgroundPriority) //ignore anything below kBackgroundPriority
					continue;
				// go through each output, record requested values
				for(uint output=0; output<NumOutputs; output++) {
					if(mc->isUsingOutput(output)) {
						cmdstatelist_t& curstatelist=cmdstates[output];
						cmdstatelist_t::index_t ent=curstatelist.begin(); // insertion sort on priority
						for(; ent!=curstatelist.end(); ent=curstatelist.next(ent))
							if(curstatelist[ent].priority<p)
								break;
						cmdstatelist_t::index_t added=curstatelist.new_before(ent);
						curstatelist[added].init(output,*it,mc,p);
					}
				}
				checkinMotion(*it); // release lock, done with motion
				// remove id from list of unprocessed motioncommands
				list<MC_ID>::iterator rem=it++;
				unlocked.erase(rem);
			}
		}
	}

	// now we've got, for each output, a list of requested values sorted by priority
	// summarize each output
	for(uint frame=0; frame<NumFrames; frame++)
		for(uint output=0; output<NumOutputs; output++) {
			cmdstatelist_t& curstatelist=cmdstates[output];
			float alpha=1;
			JointCmd sumcmd;
			cmdstatelist_t::index_t ent=curstatelist.begin();
			while(ent!=curstatelist.end() && alpha>0) {
				JointCmd curcmd;
				float curp=curstatelist[ent].priority;
				float curalpha=1; // curalpha is multiplicative sum of leftovers (weights between 0 and 1)
				for(;curstatelist[ent].priority==curp; ent=curstatelist.next(ent)) {
					//weighted average within priority level
					float curweight=curstatelist[ent].frames[frame].weight;
					ASSERT(curweight>=0,"negative output weights are illegal")
					if(curweight<0) //negative weights are illegal
						curweight=0;
					curcmd.value+=curstatelist[ent].frames[frame].value*curweight;
					curcmd.weight+=curweight;
					if(curweight<1)
						curalpha*=(1-curweight);
					else
						curalpha=0;
				}
				if(curcmd.weight>0) {
					//weighted average of priority levels
					sumcmd.value+=curcmd.value/curcmd.weight*alpha;
					sumcmd.weight+=alpha;
					alpha*=curalpha;
				}
			}
			if(sumcmd.weight>0) 
				outputs[frame][output]=sumcmd.value/sumcmd.weight;
			else //if zero weight, hold last value
				outputs[frame][output]=cmdSums[output];
			if(frame==NumFrames-1)
				cmds[output]=sumcmd;
		}
	
	for(uint output=0; output<NumOutputs; output++)
		cmdSums[output]=outputs[NumFrames-1][output];
				
	// now summarize each output's PID values (for those which use PID control)
	for(uint output=PIDJointOffset; output<PIDJointOffset+NumPIDJoints; output++) {
		cmdstatelist_t& curstatelist=cmdstates[output];
		float alpha=1;
		float sumpid[3];
		for(uint i=0; i<3; i++)
			sumpid[i]=0;
		float sumweight=0;
		cmdstatelist_t::index_t ent=curstatelist.begin();
		while(ent!=curstatelist.end() && alpha>0) {
			float tmppid[3];
			for(uint i=0; i<3; i++)
				tmppid[i]=0;
			float tmpweight=0;
			float curp=curstatelist[ent].priority;
			float curalpha=1; // curalpha is multiplicative sum of leftovers (weights between 0 and 1)
			for(;curstatelist[ent].priority==curp; ent=curstatelist.next(ent)) {
				//weighted average within priority level
				float curweight=curstatelist[ent].pid_weight;
				ASSERT(curweight>=0,"negative PID weights are illegal")
				if(curweight<0) //negative weights are illegal
					curweight=0;
				for(uint i=0; i<3; i++)
					tmppid[i]+=curstatelist[ent].pid[i]*curweight;
				tmpweight+=curweight;
				if(curweight<1)
					curalpha*=(1-curweight);
				else
					curalpha=0;
			}
			if(tmpweight>0) {
				//weighted average of priority levels
				for(uint i=0; i<3; i++)
					sumpid[i]+=tmppid[i]/tmpweight*alpha;
				sumweight+=alpha;
				alpha*=curalpha;
			}
		}
		if(sumweight>0) {
			for(uint i=0; i<3; i++)
				sumpid[i]/=sumweight;
			setPID(output,sumpid);
		}
	}

	func_end();
	//	if(begin(id)!=end())
	//if(state && state->buttons[LFrPawOffset]) cout << "getAngles-done." << flush;
}

void
MotionManager::updateWorldState() {
	for(uint output=LEDOffset; output<LEDOffset+NumLEDs; output++)
		state->outputs[output]=cmdSums[output];
	for(uint output=EarOffset; output<EarOffset+NumEarJoints; output++)
		state->outputs[output]=cmdSums[output];
}

bool
MotionManager::updatePIDs(OPrimitiveID primIDs[NumOutputs]) {
	bool dirty=!pidchanges.empty();
	while(!pidchanges.empty()) {
		double gain[3];
		word shift[3];
		for(uint i=0; i<3; i++) {
			gain[i]=pidchanges.front().pids[i]*2;
			shift[i]=0xF;
			while(shift[i]!=0 && gain[i]!=(int)gain[i] && gain[i]<(1<<15)) {
				gain[i]*=2;
				shift[i]--;
			}
		}
		OPENR::SetJointGain(primIDs[pidchanges.front().joint],(word)gain[0],(word)gain[1],(word)gain[2],shift[0],shift[1],shift[2]);
		for(uint i=0; i<3; i++)
			state->pids[pidchanges.front().joint][i]=pidchanges.front().pids[i];
		pidchanges.pop_front();
	}
	return dirty;
}

// documentation for this function is at the end of the file
MotionManager::MC_ID
MotionManager::addMotion(const SharedMotionBase& sm) {
	//	cout << "addMotion..." << flush;
	while(numAcc<MAX_ACCESS-1) { cout << "WAIT" << flush; } //Wait for everyone to register
	accID_t id = func_begin();
	//	cout << id << "..." << flush;
	MC_ID mc_id = pop_free();
	if(mc_id==cmdlist.end())
		return func_end(cmdlist.end());
	//	cout << mc_id << "..." << flush;
	cmdlist[mc_id].baseaddrs[id]=(MotionCommand*)sm.DataBase();
	//	cout << cmdlist[mc_id].baseaddrs[id] << "..." << flush;
	cmdlist[mc_id].rcr[id]=sm.getRegion();
	cmdlist[mc_id].lastAccessor=id;
	sm.MotionManagerMsgBase()->setAdd(mc_id);
	OStatus err;
	ASSERT((err=subjs[id]->SetData(sm.getRegion()))==oSUCCESS,"*** ERROR MotionManager: SetData returned " << err);
	ASSERT((err=subjs[id]->NotifyObservers())==oSUCCESS,"*** ERROR MotionManager: NotifyObservers returned " << err);
	//	cout << "addMotion-done" << endl;
	return func_end(mc_id);
}

void
MotionManager::receivedMsg(const ONotifyEvent& event) {
	//	cout << "receivedMsg..." << flush;
	accID_t id = func_begin();
	//	cout << id << "..." << flush;
	for(int x=0; x<event.NumOfData(); x++) {
		RCRegion * rcr = event.RCData(x);
		MotionManagerMsg * mminfo = (MotionManagerMsg*)rcr->Base();
		switch(mminfo->type) {
		case MotionManagerMsg::addMotion: {
			//			rcr = new RCRegion(rcr->Header()); // This is a little funky, but that's the way robosoccer does it...
			//!@test i'm not sure this will work - RCRegion::Header is private now...
			rcr->AddReference();
			MC_ID mc_id=mminfo->mc_id;
			cmdlist[mc_id].rcr[id]=rcr;
			cmdlist[mc_id].baseaddrs[id]=(MotionCommand*)(rcr->Base()+MotionManagerMsg::SIZEOF_MSG);
		} break;
		case MotionManagerMsg::deleteMotion: {
			MC_ID mc_id=mminfo->mc_id;
			cmdlist[mc_id].rcr[id]->RemoveReference();
		} break;
		default:
			printf("*** WARNING *** unknown MotionManager msg type received\n");
		}
	}
	//	cout << "receivedMsg-done" << endl;
	func_end();
}

MotionCommand *
MotionManager::checkoutMotion(MC_ID mcid,bool block,accID_t id) {
	//cout << "checkout..." << flush;
	if(mcid>=MAX_MOTIONS) {
		cout << "*** WARNING *** " << id << " tried to access invalid mcid " << mcid << endl;
		return NULL;
	}
	if(block)
		cmdlist[mcid].lock.lock(id);
	else
		if(!cmdlist[mcid].lock.try_lock(id))
			return NULL;
	if(cmdlist[mcid].lastAccessor==(accID_t)-1) {
		cout << "*** WARNING *** " << id << " tried to access dead mcid " << mcid << endl;
		cmdlist[mcid].lock.release();
		return NULL;
	}
	//cout << "locked..." << endl;
	MotionCommand * base = cmdlist[mcid].baseaddrs[id];
	//	cout << "base=" << base << "..." << flush;
	if(cmdlist[mcid].lastAccessor!=id) {
		//cout << "converting from " << MCRegistrar::getRaw(base) << "..." << flush;
		//cout << "prev=" << accRegs[cmdlist[mcid].lastAccessor].getReg(base) << "..." << flush;
		//		accRegs[id].convert(base);
		//cout << "to=" << MCRegistrar::getRaw(base) << ", " << accRegs[cmdlist[mcid].lastAccessor].getReg(base) << endl;
		cmdlist[mcid].lastAccessor=id;
	}
	//cout << "checkout-done..." << flush;
	return base;
}

void
MotionManager::checkinMotion(MC_ID mcid) {
	cmdlist[mcid].lock.release();
}

//! @todo i don't think i'm handling the AddRefence/RemoveReference stuff right!
void
MotionManager::removeMotion(MC_ID mcid) {
	accID_t id = func_begin();
	checkoutMotion(mcid,true,id);
	MotionManagerMsg dmsg;
	dmsg.setDelete(mcid);
	subjs[id]->SetData(&dmsg,sizeof(dmsg));
	subjs[id]->NotifyObservers();
	cmdlist[mcid].rcr[id]->RemoveReference();
	push_free(mcid);
	checkinMotion(mcid);
	func_end();
}

uint
MotionManager::cleanup(accID_t id) {
	uint cleaned=0;
	//	accID_t id = func_begin();
	MC_ID it=begin(id);
	while(it!=end()) {
		MotionCommand * curMC = checkoutMotion(it,true,id);
		if(curMC->shouldPrune()) {
			cout << "Removing expired" << endl;
			MC_ID tmp=next(it,id);
			removeMotion(it);
			cleaned++;
			checkinMotion(it);
			it=tmp;
		} else {
			checkinMotion(it);
			it=next(it,id);
		}
	}
	//	func_end();
	return cleaned;
}

void
MotionManager::OutputState::init(unsigned int output, MC_ID mc_id, MotionCommand* mc, float p) {
	priority=p;
	mcid=mc_id;
	mc->planJointCmds(output, frames);
  if(output+1>PIDJointOffset && output<PIDJointOffset+NumPIDJoints) {
		pid_weight=1;
		float * tmp=mc->getPIDValue(output, pid_weight);
		if(tmp==NULL)
			for(uint i=0; i<3; i++)
				pid[i]=DefaultPIDs[output][i];
		else
			for(uint i=0; i<3; i++)
				pid[i]=tmp[i];
	}
}

/*! Note that we don't actually set the PIDs in the system here, we just queue them up.
 *  PID changes seem to be an expensive operation, so may only want to clear the queue
 *  at some reduced rate (although that's not actually currently implemented, so right
 *  now there's no real benefit until that's done) */
void
MotionManager::setPID(unsigned int joint, const float pids[3]) {
	func_begin();
	//see if there's already an update for this joint
	for(uint u = pidchanges.begin(); u!=pidchanges.end(); u=pidchanges.next(u)) {
		if(pidchanges[u].joint==joint) { //found it
			for(uint i=0; i<3; i++) {
				pidchanges[i].pids[i]=pids[i];
				if(pids[i]!=state->pids[joint][i]) { //see if we're setting back to current PID
					for(i++; i<3; i++) //we aren't, copy over the rest
						pidchanges[i].pids[i]=pids[i];
					func_end();
					return;
				}
			}
			//if it didn't return within the loop, no difference was found from current state
			//so just delete the update
			pidchanges.erase(u);
			func_end();
			return;
		}
	}
	//if we didn't return from the loop, we didn't find an update for the joint
	for(uint i=0; i<3; i++) //check to see if it's different from the current
		if(pids[i]!=state->pids[joint][i]) {
			PIDUpdate update(joint,pids); //it is different, insert a new update
			pidchanges.push_back(update);
			break;
		}
	func_end();
}


MotionManager::accID_t
MotionManager::func_begin() {
	MMlock.lock(_MMaccID);
	return _MMaccID;
}

void
MotionManager::func_end() {
	MMlock.release();
}

MotionManager::MC_ID
MotionManager::skip_ahead(MC_ID mcid, accID_t acc) const {
	// this is in case a new motion has been added, but the current
	// process hasn't received its own copy yet, so should skip over them
	while(mcid!=cmdlist.end() && cmdlist[mcid].rcr[acc]==NULL)
		mcid=cmdlist.next(mcid);
	return mcid;
}

/*! @file
 * @brief Implements MotionManager, simplifies sharing of MotionCommand's and provides mutual exclusion to their access
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name:  $
 * $Revision: 1.12 $
 * $State: Exp $
 * $Date: 2003/02/24 10:34:56 $
 */


/*
		for(uint f=0;f<NumFrames;f++)
			for(uint i=0; i<NumOutputs; i++)
				outputs[f][i]=0;
		const uint cyctime=128;
		uint ot=get_time()+3*cyctime;
		for(uint f=0;f<NumFrames;f++) {
			uint t=ot+f*FrameTime;
			outputs[f][TopBrLEDOffset]=(t/(double)cyctime-t/cyctime)*.75+.25;
			t-=cyctime;
			outputs[f][TopLLEDOffset]=(t/(double)cyctime-t/cyctime)*.75+.25;
			outputs[f][TopRLEDOffset]=(t/(double)cyctime-t/cyctime)*.75+.25;
			t-=cyctime;
			outputs[f][MidLLEDOffset]=(t/(double)cyctime-t/cyctime)*.75+.25;
			outputs[f][MidRLEDOffset]=(t/(double)cyctime-t/cyctime)*.75+.25;
			t-=cyctime;
			outputs[f][BotLLEDOffset]=(t/(double)cyctime-t/cyctime)*.75+.25;
			outputs[f][BotRLEDOffset]=(t/(double)cyctime-t/cyctime)*.75+.25;
		}
*/
