#ifndef INCLUDED_MotionCommand_h
#define INCLUDED_MotionCommand_h

#include "JointCmd.h"
#include "Shared/RobotInfo.h"

//! The abstract base class for motions, provides common interface.  All motions should inherit from this
/*! For instructions on how to create:
 * - <b>a new subclass</b> of MotionCommand, read on.  Also see the step-by-step
 *   <a href="../FirstMotionCommand.html">guide</a>
 * - <b>an instantiation</b> of a MotionCommand subclass, see MotionManager
 * 
 * To create a new type of motion, you'll want to subclass this.  You don't need to do
 * anything fancy, but just be sure to override the 4 abstract functions.
 *
 * When a PID or Output is set to a value, that value is retained until it is set to a new value,
 * even if the MotionCommand that set it is pruned or stops using the output.  Outputs never
 * "reset" to 0 or some other relatively arbitrary base value.
 *
 * Be aware that there is a delay between when you set a joint to a value and that actually
 * is taken into account by the system - it's on the order of RobotInfo::FrameTime*RobotInfo::NumFrames
 * (currently 8*4 = 32 ms, at most 2*8*4 = 64 ms) This is because the commands are double buffered.
 * PIDs, on the other hand, seem to take effect more quickly.  This un-synchronization can sometimes
 * cause annoying jerkiness (mainly on startup, where there's a large difference between desired
 * and target values.)
 *
 * Here is the cycle of calls made by MotionManager to your command:
 * -# shouldPrune() (by default, this will call isAlive() iff autoprune==true)
 * -# updateJointCmds() (assuming the MC wasn't pruned after the previous step)
 * -# getPriority()
 * -# once for each output:
 *    -# isUsingOutput()
 *    -# if isUsingOutput() returned true, then:
 *       -# planJointCmds() (which by default simply replicates a call to getJointCmd() )
 *       -# PID controlled joints only: getPIDValue()
 * 
 *  Notice that the results of getPriority() and weight returned by the getJointCmd() will
 * affect the expression of the next ::NumFrames returned by planJointCmds as a block.
 * Weights within those ::NumFrames JointCmds() will be applied frame by frame\n
 * @warning <b>Don't store pointers in motion commands!</b> \n
 * Since motion commands are in shared memory, and these shared memory regions can have different
 * base pointers in each process, pointers will only be valid in the process from which they were
 * assigned.  In other processes, that address may point to something else, especially if it was
 * pointing outside of the shared memory regions.\n
 *  There are convoluted ways of getting around this.  If needed, MotionManager could be modified
 * to hand out shared memory regions upon request.  Let's try to avoid this for now.  Keep MotionCommands
 * simple, without dynamic memory.  Do more complicated stuff with behaviors, which only have to run in Main.
 * @see REGDEF @see REGIMP
 */
class MotionCommand {
	//!@nosubgrouping

 public:

	//! General guidelines for priority values
	/*! Note that you can add and subtract these values, and priorities are double values, so you can always define an ordering */
	enum StdPriorityLvl_t {
		kIgnoredPriority    =-1,  //!< won't be expressed, handy if you want to temporarily pause something
		kBackgroundPriority = 0,  //!< will only be expressed if *nothing* else is using that joint
		kLowPriority        = 5,  //!< for stuff that's not background but lower than standard
		kStdPriority        = 10, //!< for every-day commands
		kHighPriority       = 50, //!< for stuff that should over ride standard stuff
		kEmergencyPriority  = 100 //!< for really important stuff, such as the emergency stop
	};
	



	//        *****************
	//! @name *** ABSTRACT: *** (must be defined by subclasses)
	//        *****************

	//! is called once per update cycle, can do any processing you need to change your priorities or weights
	/*! @return zero if no changes were made, non-zero otherwise @see RobotInfo::NumFrames @see RobotInfo::FrameTime */
	virtual int updateJointCmds()=0;

	//! gets the value that a output should be set to
	/*! this may be called often - use a cache, which gets updated by a call to updateJointCmds()\n
	 *  joint angles are in radians (not microradians), booleans are zero/non-zero
	 *  @param i joint/led ID number (from RobotInfo.h, aka position in PID list)
	 *  @return the JointCmd for that joint*/
	virtual const JointCmd& getJointCmd(unsigned int i)=0;

	
	//! not used by MotionManager at the moment, but could be used to reduce recomputation, and you may find it useful
	/*! @return zero if none of the commands have changed since last getJointCmd(), else non-zero */
	virtual int isDirty()=0;

	//! used to prune "dead" motions from the MotionManager
	/*! note that a motion could be "paused" or inactive and therefore not dirty, 
	 * but still alive, biding its time to "strike" ;)
	 * @return zero if the motion is still processing, non-zero otherwise */
	virtual int isAlive()=0;

	//@}

	
	//        ******************
	//! @name *** INHERITED: ***
	//        ******************

	public:
	//! Constructor: Defaults to kStdPriority and autoprune==true
	MotionCommand() : priority(kStdPriority), autoprune(true) {}
	//! Destructor
	virtual ~MotionCommand() {}

	//! Called to plan ahead frames (because several frames are sent together) see RobotInfo::NumFrames
	/*! This call is only made if the joint in question is going to be expressed
	 *  @param i output being requested
	 *  @param frames the array to be filled in
	 *  @see RobotInfo::NumFrames */
	virtual void planJointCmds(unsigned int i, JointCmd frames[NumFrames]) {
		frames[0]=getJointCmd(i);
		for(unsigned int f=1; f<NumFrames; f++)
			frames[f]=frames[f-1];
	}

	//! Called to see if a joint is even used before bothering to loop over frames
	/*! You might want to over ride this if you do something mid-buffer and need finer control.\n
	 *  Otherwise, this is just going to use the weight of getJointCmd(i).
	 *  @param i output being queried @return @c getJointCmd(i).weight */
	virtual bool isUsingOutput(unsigned int i) {
		return getJointCmd(i).weight>0;
	}

	//! If you returned true to isUsingOutput(i), this will be called to record the requested PID value (see RobotInfo::DefaultPIDs)
	/*! The first parameter is the output index, the second should be set to the weight you want to use\n
	 *  @return NULL if you just want to use the associated DefaultPIDs[] value, an array of double[3] otherwise, in order PID */
	virtual float* getPIDValue(unsigned int, float&) { return NULL; }

	/*! @return double representing priority level of motion */
	virtual double getPriority() { return priority; }
	
	/*! @param p double representing requested priority level
	 *  @return a reference to @a this */
	virtual MotionCommand& setPriority(double p) { priority=p; return *this; }

	/*! @return current setting of autopruning - used to remove motion from groups when !isAlive() */
	virtual bool getAutoPrune() { return autoprune; }

	/*! @param ap bool representing requested autopruning setting */
	virtual void setAutoPrune(bool ap) { autoprune=ap; }

	//! whether this motion should be removed from its motion group automatically ( MotionCommand::autoprune && !isAlive())
	/*! @return (MotionCommand::autoprune && !isAlive())*/
	virtual bool shouldPrune() { return (autoprune && !isAlive()); }

 protected:
	//! this utility function will probably be of use to a lot of MotionCommand's
	/*! Does a weighted average of a and b, favoring b by x percent (so x==0 results in a, x==1 results in b)
	 * @param a first value
	 * @param b second value
	 * @param x weight of second value as opposed to first
	 * @return @f$a*(1.0-x)+b*x@f$
	 * @todo - replace with a more fancy spline based thing? */
	static inline double interpolate(double a, double b, double x) {
		return a*(1.0-x)+b*x;
	}
	//! this utility function will probably be of use to a lot of MotionCommand's, see interpolate(double a,double b,double r)
	/*! interpolates both value and weights of JointCmd's
	 *  @param a first joint cmd
	 *  @param b second joint cmd
	 *  @param x weight to favor b's value and weight
	 *  @param r joint cmd to store the result */
	static inline void interpolate(const JointCmd& a, const JointCmd& b, double x, JointCmd& r){
		r.set(interpolate(a.value,b.value,x),interpolate(a.weight,b.weight,x));
	}
	//@}

	double priority; //!< priority level, @see StdPriorityLvl_t
	int autoprune; //!< default true, autoprune setting, if this is true and isAlive() returns false, MotionManager will attempt to remove the MC automatically
	//	REGDEF(MotionCommand,Registerable,priority,autoprune); //!<implements Registerable, REGIMP in MCRegistrar.cc
};

/*! @file
 * @brief Defines the MotionCommand class, used for creating motions of arbitrary complexity
 * @author ejt (Creator)
 *
 * $Author: ejt $
 * $Name:  $
 * $Revision: 1.9 $
 * $State: Exp $
 * $Date: 2003/02/24 10:34:56 $
 */

#endif

