//-*-c++-*-
#ifndef MMCombo_h_DEFINED
#define MMCombo_h_DEFINED

#include "Shared/debuget.h"
#include "Shared/RobotInfo.h"
#include "Wireless/Wireless.h"

#include <OPENR/OObject.h>
#include <OPENR/OSubject.h>
#include <OPENR/OObserver.h>
#include "def.h"

class BehaviorBase; // defined elsewhere, don't need whole header file here

//! Contains code for both MainObj and MotoObj processes
/*! Why go to all this trouble?  Virtual functions and polymorphism!  Instead
 *  of writing my own object typing and serialization system, I would rather
 *  just use C++'s.  But function lookups of the run time type information (RTTI)
 *  will break unless the object that created the object and the object that's
 *  actually calling the function agree on what object A's information is.
 *
 *  The easiest way to guarantee this is to compile them as one object, and
 *  then replace the strings in the source binary with strings for each of
 *  the final objects so they'll each have their own identity, but share
 *  the same code.
 *
 *  This is as close as I can get to a "fork", which is what i really want.
 */
class MMCombo : public OObject {
public:
	//! constructor
	MMCombo();
	virtual ~MMCombo() {} //!< destructor
	
	OSubject*    subject[numOfSubject];   //!< holds information for each of our subjects (data we provide)
	OObserver*   observer[numOfObserver]; //!< holds information for each of the sources we're observing
	
	virtual OStatus DoInit   (const OSystemEvent&);      //!< first call (after constructor), set up memory
	virtual OStatus DoStart  (const OSystemEvent&);      //!< second call, ask for messages
	virtual OStatus DoStop   (const OSystemEvent&);      //!< next to last call, stop sending and receiving messages
	virtual OStatus DoDestroy(const OSystemEvent&);      //!< last call (before destructor), clean up memory here
	
	void ReadyRegisterWorldState(const OReadyEvent&);    //!< main only, send out the state global
	void GotWorldState(const ONotifyEvent& event);       //!< motion only, called when state global is received
	void ReadyRegisterMotionManager(const OReadyEvent&); //!< motion only, send out motman global
	void GotMotionManager(const ONotifyEvent& event);    //!< main only, called when motman global is received
	
	void ReadySendJoints(const OReadyEvent& event);      //!< motion only (until main does ears again, then both) calls SendJoints, if DoStart has already been called
	void GotSensorFrame(const ONotifyEvent& event);      //!< main only, called when new sensor information is available
	void GotImage(const ONotifyEvent& event);            //!< main only, called when a new image is available
	void GotPowerEvent(void * msg);                      //!< main only, called when a power event occurs (can be just status events)
	
	void GotMotionMsg(const ONotifyEvent& event);        //!< both, called when a new MotionManagerMsg has been received

  void ListenCont (void* msg) { wireless->ListenCont(msg); }
  void BindCont   (void* msg) { wireless->BindCont(msg); }
  void ConnectCont(void* msg) { wireless->ConnectCont(msg); }
  void SendCont   (void* msg) { wireless->SendCont(msg); }
  void ReceiveCont(void* msg) { wireless->ReceiveCont(msg); }
  void CloseCont  (void* msg) { wireless->CloseCont(msg); }
  
protected:
	void OpenPrimitives();                               //!< both, called from SetupOutputs() (mostly for motion, but main does ears), uses #open to tell which to open
	void SetupOutputs(const bool to_open[NumOutputs]);   //!< both, called from DoInit() (mostly for motion, but main does ears)
	RCRegion* InitRegion(unsigned int size);             //!< both, called to set up a shared memory region of a given size

	BehaviorBase * startup; //!< the StartupBehavior, launched once the dog is booted, stopped when it's shutting down

  RCRegion * motmanMemRgn;     //!< Motion creates, Main receives
  RCRegion * worldStateMemRgn; //!< Main creates, Motion receives

	OPrimitiveID primIDs[NumOutputs];    //!< both, Main ears only, Motion the rest
	static const unsigned int NUM_COMMAND_VECTOR=2; //!< both, for double buffering
	RCRegion*    region[NUM_COMMAND_VECTOR]; //!< both, the actual buffers

	double  ledActivation[NumLEDs]; //!< Motion, used for partial LED activation

	unsigned int runLevel;           //!< Main, incremented until all sections are ready
	static const unsigned int readyLevel=3; //!< Main, runLevel at which StartBehavior is created. (1 for power, 1 for sensor, 1 for shared memory)
	void addRunLevel();              //!< Main, checks runLevel and creates StartBehavior when ready

	bool open[NumOutputs]; //!< motion only (until main does ears again) holds information regarding which outputs are open in ("controlled by") this process

	//! Motion only, maintains the activation level of the LEDs, returns whether it should be 'fired'
	inline OLEDValue calcLEDValue(unsigned int i,double x) {
		if(x<=0.0) {
			return oledOFF;
		} else if(x>=1.0) {
			return oledON;
		} else {
			ledActivation[i]+=x;
			if(ledActivation[i]>=1.0) {
				ledActivation[i]-=1.0;
				return oledON;
			} else {
				return oledOFF;
			}						
		}
	}

private:
//	WorldStateSerializer* wstateserializer;
	MMCombo(const MMCombo&); //!< should never be called...
	MMCombo& operator=(const MMCombo&); //!< should never be called...
};

/*! @file
 * @brief Describes MMCombo, the OObject which "forks" (sort of) into Main and Motion processes
 * @author ejt (Creator)
 *
 * $Author: alokl $
 * $Name:  $
 * $Revision: 1.11 $
 * $State: Exp $
 * $Date: 2003/02/20 19:33:32 $
 */

#endif





